<?php

namespace Myvetshop\Module\Clinique\Presenter;

use Myvetshop\Module\Clinique\Api\Crokit\Dto\CrokitRecommandation;
use Myvetshop\Module\Clinique\Entity\Recommandation;

class RecommandationPresenter
{
    private AnimalPresenter $animalPresenter;

    private RecommandationProductPresenter $recommandationProductPresenter;

    public function __construct(
        AnimalPresenter $animalPresenter,
        RecommandationProductPresenter $recommandationProductPresenter
    ) {
        $this->animalPresenter = $animalPresenter;
        $this->recommandationProductPresenter = $recommandationProductPresenter;
    }

    /**
     * @return array{
     *     id: int,
     *     codeReco: string,
     *     dateReco: \DateTimeImmutable,
     *     espece: string,
     *     race: string,
     *     idRecommandationProduct: int|null,
     *     animal: array{
     *          id: int,
     *          id_customer: int,
     *          espece: string|null,
     *          race: string|null,
     *          id_gender: int,
     *          robe: string|null,
     *          nom: string,
     *     },
     *     products: list<array{
     *          id: int,
     *          months: int|null,
     *          quantity: int|null,
     *          product: \Product|null,
     *          image_id: int|null,
     *     }>,
     *     cartRule: \CartRule|null,
     * }
     */
    public function present(Recommandation $recommandation): array
    {
        return [
            'id' => $recommandation->getId() ?? 0,
            'codeReco' => $recommandation->codeReco ?? '',
            'dateReco' => $recommandation->dateReco,
            'espece' => $recommandation->espece,
            'race' => $recommandation->race,
            'animal' => $this->animalPresenter->present($recommandation->animal),
            'idRecommandationProduct' => $recommandation->idRecommandationProduct,
            'products' => \array_values(
                \array_map(
                    [$this->recommandationProductPresenter, 'present'],
                    $recommandation->getProduits()->toArray(),
                )
            ),
            'cartRule' => null,
        ];
    }

    /**
     * @return array{
     *     codeReco: string,
     *     dateReco: \DateTimeImmutable,
     *     espece: string,
     *     race: string,
     *     animal: array{
     *           nom: string
     *     },
     * }
     */
    public function presentDto(CrokitRecommandation $crokitRecommandation): array
    {
        return [
            'codeReco' => $crokitRecommandation->codeReco,
            'dateReco' => $crokitRecommandation->date,
            'espece' => $crokitRecommandation->esp,
            'race' => $crokitRecommandation->race,
            'animal' => [
                'nom' => $crokitRecommandation->petname,
            ],
        ];
    }

    /**
     * @return array{
     *     id: int|null,
     *     id_animal: int|null,
     *     code_reco: string,
     *     date_reco: string,
     *     petname: string,
     *     espece: string,
     *     race: string,
     *     id_gender: int,
     *     activite: string,
     *     age: string,
     *     stade_physio: string,
     *     pathologie: string|null,
     *     poids_actuel: string,
     *     poids_ideal: string,
     *     be: int|null,
     *     nbr_aliment: int,
     *     product_id_mvs1: int,
     *     product_id_mvs2: int|null,
     *     product_id_mvs3: int|null,
     *     qte_aliment1: int,
     *     qte_aliment2: int|null,
     *     qte_aliment3: int|null,
     *     transition12: int|null,
     *     transition23: int|null,
     *     date_add: string,
     *     date_upd: string,
     *     discount_code: string|null
     * }
     */
    public function presentAdmin(Recommandation $recommandation): array
    {
        return [
            'id' => $recommandation->getId() ?? null,
            'id_animal' => $recommandation->animal->getId() ?? null,
            'code_reco' => $recommandation->codeReco,
            'date_reco' => $recommandation->dateReco->format('Y-m-d'),
            'petname' => $recommandation->animal->nom,
            'espece' => $recommandation->espece,
            'race' => $recommandation->race,
            'id_gender' => $recommandation->idGender,
            'activite' => $recommandation->activite,
            'age' => $recommandation->age,
            'stade_physio' => $recommandation->stadePhysio,
            'pathologie' => $recommandation->pathologie,
            'poids_actuel' => $recommandation->poidsActuel,
            'poids_ideal' => $recommandation->poidsIdeal,
            'be' => $recommandation->be,
            'nbr_aliment' => $recommandation->nbrAliment,
            'product_id_mvs1' => $recommandation->productIdMvs1,
            'product_id_mvs2' => $recommandation->productIdMvs2,
            'product_id_mvs3' => $recommandation->productIdMvs3,
            'qte_aliment1' => $recommandation->qteAliment1,
            'qte_aliment2' => $recommandation->qteAliment2,
            'qte_aliment3' => $recommandation->qteAliment3,
            'transition12' => $recommandation->transition12,
            'transition23' => $recommandation->transition23,
            'date_add' => $recommandation->getDateAdd()->format('Y-m-d'),
            'date_upd' => $recommandation->getDateUpd()->format('Y-m-d'),
            'discount_code' => null,
        ];
    }
}
