<?php

namespace Myvetshop\Module\Clinique\Controller;

use Myvetshop\Module\Clinique\Accounting\Export\AccountingExporter;
use Myvetshop\Module\Clinique\Form\Admin\Accounting\DownloadExportType;
use PrestaShop\PrestaShop\Core\Form\FormHandlerInterface;
use PrestaShopBundle\Controller\Admin\FrameworkBundleAdminController;
use PrestaShopBundle\Security\Annotation\AdminSecurity;
use Symfony\Component\Form\FormFactoryInterface;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\StreamedResponse;

class AccountingController extends FrameworkBundleAdminController
{
    protected AccountingExporter $exporter;

    protected FormFactoryInterface $formFactory;

    protected FormHandlerInterface $exportFormHandler;

    /**
     * @param array{date_from?: string, date_to?: string} $formData
     *
     * @return FormInterface<array{date_from: string, date_to: string}>
     */
    protected function getDownloadForm(array $formData): FormInterface
    {
        return $this->formFactory
            ->create(
                DownloadExportType::class,
                $formData,
                [
                    'action' => $this->generateUrl('accounting_export_download'),
                ]
            );
    }

    public function __construct(
        AccountingExporter $exporter,
        FormFactoryInterface $formFactory,
        FormHandlerInterface $exportFormHandler
    ) {
        $this->exporter = $exporter;
        $this->exportFormHandler = $exportFormHandler;
        $this->formFactory = $formFactory;

        parent::__construct();
    }

    /**
     * @AdminSecurity("is_granted(['read'], request.get('_legacy_controller'))", message="You do not have permission to get accouting exports.")
     */
    public function exportAction(Request $request): ?Response
    {
        $form = $this->exportFormHandler->getForm();
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var array{date_from: string, date_to: string} $data */
            $data = $form->getData();

            $startDate = \DateTimeImmutable::createFromFormat('Y-m-d', $data['date_from']);
            $endDate = \DateTimeImmutable::createFromFormat('Y-m-d', $data['date_to']);

            \assert($startDate instanceof \DateTimeImmutable);
            \assert($endDate instanceof \DateTimeImmutable);

            $stats = $this->exporter->getStats($startDate, $endDate);

            $downloadForm = $this->getDownloadForm(
                [
                    'date_from' => $startDate->format('Y-m-d'),
                    'date_to' => $endDate->format('Y-m-d'),
                ]
            );
        } else {
            $stats = null;
            $startDate = null;
            $endDate = null;
            $downloadForm = null;
        }

        return $this->render(
            '@Modules/myvetshopclinique/templates/admin/accounting/export.html.twig',
            [
                'exportForm' => $form->createView(),
                'stats' => $stats,
                'startDate' => $startDate,
                'endDate' => $endDate,
                'downloadForm' => $downloadForm ? $downloadForm->createView() : null,
            ]
        );
    }

    /**
     * @AdminSecurity("is_granted(['read'], request.get('_legacy_controller'))", message="You do not have permission to get accouting exports.")
     */
    public function downloadExportAction(Request $request): Response
    {
        $downloadForm = $this->getDownloadForm([]);
        $downloadForm->handleRequest($request);

        if (!$downloadForm->isSubmitted()) {
            throw new \Exception('Form not submitted');
        }

        /** @var array{date_from: string, date_to: string} $data */
        $data = $downloadForm->getData();

        $startDate = \DateTimeImmutable::createFromFormat('Y-m-d', $data['date_from']);
        $endDate = \DateTimeImmutable::createFromFormat('Y-m-d', $data['date_to']);

        \assert($startDate instanceof \DateTimeImmutable);
        \assert($endDate instanceof \DateTimeImmutable);

        $csv = $this->exporter->getCsv($startDate, $endDate);
        $filename = 'export-comptable-'
            . $startDate->format('Y-m-d')
            . '-'
            . $endDate->format('Y-m-d')
            . '.csv';

        return new StreamedResponse(
            function () use ($csv) {
                fpassthru($csv);
                fclose($csv);
            },
            Response::HTTP_OK,
            [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => 'attachment; filename="' . $filename . '"',
            ]
        );
    }
}
