<?php

namespace App\Tests\A_Unit\Accounting\Export\Exporter\PostProcess;

use Myvetshop\Module\Clinique\Accounting\Export\Model\ExportLine;
use Myvetshop\Module\Clinique\Accounting\Export\PostProcess\RegroupByDayPostProcessor;
use PHPUnit\Framework\TestCase;

class RegroupByDayPostProcessorTest extends TestCase
{
    /**
     * @return list<array{0: string, 1: ExportLine}>
     */
    public function provideComputeLineRegroupKeyData(): array
    {
        return [
            // Usecase 1 : RegroupableLine
            [
                '/^2024-01-01-Vente internet-Axepta BPN Paribas-411000000-4110001$/',
                new ExportLine(
                    new \DateTimeImmutable('2024-01-01'),
                    'VE',
                    '411000000',
                    '4110001',
                    'Vente internet',
                    '644304',
                    77.99,
                    0.0,
                    'Axepta BPN Paribas'
                ),
            ],
            // Usecase 2 : Line not regrouped (refund)
            [
                '/^2024-01-01-Remboursement internet CB-Axepta BPN Paribas-411000000-4110001-694933-[0-9a-f.]+$/',
                new ExportLine(
                    new \DateTimeImmutable('2024-01-01'),
                    'VE',
                    '411000000',
                    '4110001',
                    'Remboursement internet CB',
                    '694933',
                    0.0,
                    53.91,
                    'Axepta BPN Paribas',
                ),
            ],
            // Usecase 2 : Line not regrouped (sell paid by stripe)
            [
                '/^2024-01-01-Vente internet-Card via Stripe-411000000-4110001-644304-[0-9a-f.]+$/',
                new ExportLine(
                    new \DateTimeImmutable('2024-01-01'),
                    'VE',
                    '411000000',
                    '4110001',
                    'Vente internet',
                    '644304',
                    77.99,
                    0.0,
                    'Card via Stripe'
                ),
            ],
        ];
    }

    /**
     * @dataProvider provideComputeLineRegroupKeyData
     */
    public function testComputeLineRegroupKey(string $expectedKeyPattern, ExportLine $testLine): void
    {
        $processor = new RegroupByDayPostProcessor();

        $resultKey = $processor->computeLineRegroupKey($testLine);

        $this->assertMatchesRegularExpression($expectedKeyPattern, $resultKey);
    }

    public function testComputeLineRegroupKeyNoConflictOnNonRegroupable(): void
    {
        $processor = new RegroupByDayPostProcessor();

        $testLine = new ExportLine(
            new \DateTimeImmutable('2024-01-01'),
            'VE',
            '411000000',
            '4110001',
            'Vente internet',
            '644304',
            77.99,
            0.0,
            'Card via Stripe'
        );

        $resultKey1 = $processor->computeLineRegroupKey($testLine);
        $resultKey2 = $processor->computeLineRegroupKey($testLine);

        $this->assertNotEquals($resultKey1, $resultKey2);
    }

    public function testAddLines(): void
    {
        $testLines = [
            new ExportLine(
                new \DateTimeImmutable('2024-01-01'),
                'VE',
                '411000000',
                '4110001',
                'Vente internet',
                '644304',
                77.99,
                0.0,
                'Axepta BPN Paribas'
            ),
            new ExportLine(
                new \DateTimeImmutable('2024-01-01'),
                'VE',
                '445711000',
                '',
                'Vente internet',
                '644304',
                0.0,
                13.0,
                'Axepta BPN Paribas'
            ),
            new ExportLine(
                new \DateTimeImmutable('2024-01-01'),
                'VE',
                '707110000',
                '',
                'Vente internet',
                '644304',
                0.0,
                64.99,
                'Axepta BPN Paribas'
            ),
            new ExportLine(
                new \DateTimeImmutable('2024-01-01'),
                'VE',
                '411000000',
                '4110001',
                'Vente internet',
                '644305',
                101.59,
                0.0,
                'Axepta BPN Paribas'
            ),
            new ExportLine(
                new \DateTimeImmutable('2024-01-01'),
                'VE',
                '445711000',
                '',
                'Vente internet',
                '644305',
                0.0,
                16.94,
                'Axepta BPN Paribas'
            ),
            new ExportLine(
                new \DateTimeImmutable('2024-01-01'),
                'VE',
                '707110000',
                '',
                'Vente internet',
                '644305',
                0.0,
                84.65,
                'Axepta BPN Paribas'
            ),
            new ExportLine(
                new \DateTimeImmutable('2024-01-01'),
                'VE',
                '411000000',
                '4110001',
                'Vente internet',
                '644306',
                42.2,
                0.0,
                'Axepta BPN Paribas'
            ),
            new ExportLine(
                new \DateTimeImmutable('2024-01-01'),
                'VE',
                '445711000',
                '',
                'Vente internet',
                '644306',
                0.0,
                7.04,
                'Axepta BPN Paribas'
            ),
            new ExportLine(
                new \DateTimeImmutable('2024-01-01'),
                'VE',
                '707110000',
                '',
                'Vente internet',
                '644306',
                0.0,
                35.16,
                'Axepta BPN Paribas'
            ),
        ];

        $processor = new RegroupByDayPostProcessor();
        $processor->addLines($testLines);

        $result = $processor->getLines();

        $this->assertCount(3, $result);
        $this->assertInstanceOf(ExportLine::class, $result[0]);
        $this->assertEquals('2024-01-01', $result[0]->getDate()->format('Y-m-d'));
        $this->assertEquals('VE', $result[0]->getJournal());
        $this->assertEquals('411000000', $result[0]->getAccount());
        $this->assertEquals('4110001', $result[0]->getAuxiliary());
        $this->assertEquals('Vente internet 01/01', $result[0]->getEntitled());
        $this->assertEquals('', $result[0]->getDocNumber());
        $this->assertEquals(221.78, $result[0]->getDebit());
        $this->assertEquals(0.0, $result[0]->getCredit());
        $this->assertEquals('Axepta BPN Paribas', $result[0]->getPaymentMethod());

        $this->assertInstanceOf(ExportLine::class, $result[1]);
        $this->assertEquals('2024-01-01', $result[1]->getDate()->format('Y-m-d'));
        $this->assertEquals('VE', $result[1]->getJournal());
        $this->assertEquals('445711000', $result[1]->getAccount());
        $this->assertEquals('', $result[1]->getAuxiliary());
        $this->assertEquals('Vente internet 01/01', $result[1]->getEntitled());
        $this->assertEquals('', $result[1]->getDocNumber());
        $this->assertEquals(0.0, $result[1]->getDebit());
        $this->assertEquals(36.98, $result[1]->getCredit());
        $this->assertEquals('Axepta BPN Paribas', $result[1]->getPaymentMethod());

        $this->assertInstanceOf(ExportLine::class, $result[2]);
        $this->assertEquals('2024-01-01', $result[2]->getDate()->format('Y-m-d'));
        $this->assertEquals('VE', $result[2]->getJournal());
        $this->assertEquals('707110000', $result[2]->getAccount());
        $this->assertEquals('', $result[2]->getAuxiliary());
        $this->assertEquals('Vente internet 01/01', $result[2]->getEntitled());
        $this->assertEquals('', $result[2]->getDocNumber());
        $this->assertEquals(0.0, $result[2]->getDebit());
        $this->assertEquals(184.8, $result[2]->getCredit());
        $this->assertEquals('Axepta BPN Paribas', $result[2]->getPaymentMethod());
    }

    public function testSumLinesRegroupable(): void
    {
        $testLines = [
            new ExportLine(
                new \DateTimeImmutable('2024-01-01'),
                'VE',
                '411000000',
                '4110001',
                'Vente internet',
                '644304',
                77.99,
                0.0,
                'Axepta BPN Paribas'
            ),
            new ExportLine(
                new \DateTimeImmutable('2024-01-01'),
                'VE',
                '411000000',
                '4110001',
                'Vente internet',
                '644305',
                101.59,
                0.0,
                'Axepta BPN Paribas'
            ),
            new ExportLine(
                new \DateTimeImmutable('2024-01-01'),
                'VE',
                '411000000',
                '4110001',
                'Vente internet',
                '644306',
                42.2,
                0.0,
                'Axepta BPN Paribas'
            ),
        ];

        $processor = new RegroupByDayPostProcessor();
        $result = $processor->sumLines($testLines);

        $this->assertInstanceOf(ExportLine::class, $result);
        $this->assertEquals('2024-01-01', $result->getDate()->format('Y-m-d'));
        $this->assertEquals('VE', $result->getJournal());
        $this->assertEquals('411000000', $result->getAccount());
        $this->assertEquals('4110001', $result->getAuxiliary());
        $this->assertEquals('Vente internet 01/01', $result->getEntitled());
        $this->assertEquals('', $result->getDocNumber());
        $this->assertEquals(221.78, $result->getDebit());
        $this->assertEquals(0.0, $result->getCredit());
        $this->assertEquals('Axepta BPN Paribas', $result->getPaymentMethod());
    }

    public function testSumLinesNotRegroupable(): void
    {
        $testLines = [
            new ExportLine(
                new \DateTimeImmutable('2024-01-01'),
                'VE',
                '411000000',
                '4110001',
                'Remboursement internet CB',
                '694933',
                0.0,
                53.91,
                'Axepta BPN Paribas',
            ),
        ];

        $processor = new RegroupByDayPostProcessor();
        $result = $processor->sumLines($testLines);

        $this->assertSame($testLines[0], $result);
    }
}
