<?php

namespace Myvetshop\Module\Clinique\Accounting\Export\Exporter\PostProcess;

use Myvetshop\Module\Clinique\Accounting\Export\Model\ExportLine;

/**
 * Correct the VAT information for 0.01ct difference to balance the export
 */
class VatRoundCorrectorPostProcessor implements ExportPostProcessorInterface
{
    public function postProcess(array $lines): array
    {
        $orderTotals = \array_reduce(
            $lines,
            function (array $totals, ExportLine $line): array {
                switch ($line->getAccount()) {
                    case '411000000':
                        $totals['paid'] += \max($line->getDebit(), $line->getCredit());
                        break;

                    case '709700000':
                        $totals['discounts'] += \max($line->getDebit(), $line->getCredit());
                        break;

                    case '445711000':
                    case '445711300':
                    case '445711100':
                        $totals['vat'] += \max($line->getDebit(), $line->getCredit());
                        break;

                    case '708000100':
                        $totals['shipping'] += \max($line->getDebit(), $line->getCredit());
                        break;

                    case '707110000':
                    case '707110100':
                    case '707110200':
                    case '707130000':
                    case '707130100':
                    case '707130200':
                        $totals['products'] = \max($line->getDebit(), $line->getCredit());
                        break;

                    default:
                        // Ignored
                }

                return $totals;
            },
            [
                'paid' => 0,
                'discounts' => 0,
                'vat' => 0,
                'products' => 0,
                'shipping' => 0,
            ]
        );

        $totalPaid = \round($orderTotals['paid'], 2);
        $totalVat = \round($orderTotals['vat'], 2);
        $totalProducts = \round($orderTotals['products'], 2);
        $totalDiscounts = \round($orderTotals['discounts'], 2);
        $totalShipping = \round($orderTotals['shipping'], 2);

        $difference = \round($totalPaid + $totalDiscounts - $totalVat - $totalProducts - $totalShipping, 2);

        // Patch the VAT amount to balance the export
        while (\abs($difference) > 0 && \abs($difference) <= 0.02) {
            // Find the first VAT line to apply the patch
            foreach ($lines as $i => $line) {
                switch ($line->getAccount()) {
                    case '445711000':
                    case '445711300':
                    case '445711100':
                        $lines[$i] = new ExportLine(
                            $line->getDate(),
                            $line->getJournal(),
                            $line->getAccount(),
                            $line->getAuxiliary(),
                            $line->getEntitled(),
                            $line->getDocNumber(),
                            $line->getDebit() > 0 ? \round($line->getDebit() + $difference, 2) : $line->getDebit(),
                            $line->getCredit() > 0 ? \round($line->getCredit() + $difference, 2) : $line->getCredit(),
                            $line->getPaymentMethod(),
                        );
                        $difference = 0;
                        break;

                    default:
                        // Ignored
                }
            }
        }

        return $lines;
    }
}
