<?php

namespace Myvetshop\Module\Clinique\Command\Clinic;

use Doctrine\ORM\EntityManagerInterface;
use Myvetshop\Module\Clinique\Entity\Factory\EstablishmentEventFactory;
use Myvetshop\Module\Clinique\Repository\MyvetshopCliniqueRepository;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

class CreateEventCommand extends Command
{
    private EntityManagerInterface $em;

    private EstablishmentEventFactory $establishmentEventFactory;

    private MyvetshopCliniqueRepository $myvetshopCliniqueRepository;

    protected function configure(): void
    {
        $this
            ->setDescription("Création manuelle d'un événement")
            ->addArgument(
                'type',
                InputArgument::REQUIRED,
                'Type d\'événement : customer_join, customer_leave, customer_update, customer_disabled, order_updated'
            )
            ->addArgument(
                'object-id',
                InputArgument::REQUIRED,
                'ID de commande ou de client (dépend on type)',
            )
            ->addArgument(
                'privilege-code',
                InputArgument::OPTIONAL,
                'Code privilège (facultatif, remplace le code par défaut)'
            );
    }

    public function __construct(
        EntityManagerInterface $em,
        EstablishmentEventFactory $establishmentEventFactory,
        MyvetshopCliniqueRepository $myvetshopCliniqueRepository
    ) {
        $this->em = $em;
        $this->establishmentEventFactory = $establishmentEventFactory;
        $this->myvetshopCliniqueRepository = $myvetshopCliniqueRepository;

        parent::__construct('app:clinic:event-create');
    }

    public function execute(InputInterface $input, OutputInterface $output): int
    {
        $styledOutput = new SymfonyStyle($input, $output);

        $type = $input->getArgument('type');
        \assert(\is_string($type));
        $objectId = \intval($input->getArgument('object-id'));

        switch ($type) {
            case 'customer_join':
            case 'customer_leave':
            case 'customer_update':
            case 'customer_disabled':
                $customer = new \Customer($objectId);
                if (!\Validate::isLoadedObject($customer)) {
                    $styledOutput->error('Impossible de charger le client');

                    return 1;
                }
                $eventObject = $this->establishmentEventFactory->createCustomerEvent($type, $customer);
                break;

            case 'order_updated':
                $order = new \Order($objectId);
                if (!\Validate::isLoadedObject($order)) {
                    $styledOutput->error('Impossible de charger la commande');

                    return 1;
                }
                $eventObject = $this->establishmentEventFactory->createOrderEvent($type, $order);
                break;

            default:
                throw new \Exception('Type non autorisé : ' . \strval($type));
        }

        $privilegeCode = $input->getArgument('privilege-code');
        if ($privilegeCode && \is_string($privilegeCode)) {
            $clinique = $this->myvetshopCliniqueRepository->findByCodePrivilege($privilegeCode);

            if (!$clinique) {
                $styledOutput->error('Code privilège non trouvé');

                return 1;
            }

            $eventObject->clinique = $clinique;
        }

        $this->em->persist($eventObject);
        $this->em->flush();

        $styledOutput->success('Événement créé avec succès');

        return 0;
    }
}
