<?php

declare(strict_types=1);

require_once __DIR__ . '/../classes/Recommandation.php';

class CricCrocApi
{
    protected function request(string $endpoint): string
    {
        $curl = curl_init();
        if (!$curl) {
            throw new Exception('Erreur initialisation cURL');
        }
        curl_setopt($curl, \CURLOPT_URL, 'https://www.crokit.fr' . $endpoint);
        curl_setopt($curl, \CURLOPT_TIMEOUT, 30);
        curl_setopt($curl, \CURLOPT_RETURNTRANSFER, 1);

        $ret = curl_exec($curl);

        if (is_bool($ret)) {
            throw new Exception('Erreur requête cURL');
        }

        return $ret;
    }

    /**
     * @return array<mixed>
     */
    public function getRecoJson(string $codeReco): array
    {
        $codeReco = strtoupper((string) $codeReco);

        if (1 !== preg_match('#^[A-Z0-9]{5}$#', $codeReco)) {
            throw new Exception('Code reco invalide. Il doit être composé de 5 lettres et chiffres.');
        }

        $ret = json_decode(
            $this->request('/MVSreturn.php?secure=SMV4521bou&codeReco=' . urlencode($codeReco)),
            true
        );

        if (is_array($ret) && isset($ret[0])) {
            return $ret[0];
        }

        throw new Exception('Retour cric-croc invalide.');
    }

    public function parseRecoJson(array $recoJson): Recommandation
    {
        if (!$recoJson['petname']) {
            throw new Exception('La recommandation n\'est pas valide.');
        }

        $ret = new Recommandation();

        $date_reco_parts = explode('-', $recoJson['date']);

        $ret->date_reco = $date_reco_parts[2] . '-' . $date_reco_parts[1] . '-' . $date_reco_parts[0];
        $ret->code_clinique = $recoJson['code'];
        $ret->petname = $recoJson['petname'];
        $ret->espece = strtolower($recoJson['esp']); // chien/chat
        $ret->race = $recoJson['race']; // chien croisé type robuste
        $ret->id_gender = str_contains(strtolower($recoJson['sex']), 'mâle') ? 1 : 2;
        $ret->activite = $recoJson['act'];
        $ret->age = $recoJson['age'];
        $ret->date_birth = empty($recoJson['dob']) ? null : $recoJson['dob'];
        $ret->stade_physio = $recoJson['stadePhysio'];
        $ret->pathologie = $recoJson['patho'];
        $ret->poids_actuel = $recoJson['actualw'];
        $ret->poids_ideal = $recoJson['idealw'];
        $ret->be = $recoJson['be'];
        $ret->nbr_aliment = $recoJson['nbrAliment'];
        $ret->product_id_mvs1 = empty($recoJson['productIDmvs1']) ? null : (int) ($recoJson['productIDmvs1']);
        $ret->product_id_mvs2 = empty($recoJson['productIDmvs2']) ? null : (int) ($recoJson['productIDmvs2']);
        $ret->product_id_mvs3 = empty($recoJson['productIDmvs3']) ? null : (int) ($recoJson['productIDmvs3']);
        $ret->qte_aliment1 = $recoJson['qteAliment1'];
        $ret->qte_aliment2 = empty($recoJson['qteAliment2']) ? null : $recoJson['qteAliment2'];
        $ret->qte_aliment3 = empty($recoJson['qteAliment3']) ? null : $recoJson['qteAliment3'];
        $ret->transition12 = empty($recoJson['transition12']) ? null : $recoJson['transition12'];
        $ret->transition23 = empty($recoJson['transition23']) ? null : $recoJson['transition23'];

        return $ret;
    }

    public function getPdf($codeReco): string
    {
        return $this->request('/pdf/' . urlencode($codeReco) . '.pdf');
    }
}
