<?php

namespace Myvetshop\Module\Clinique\Import\Kalivet\ImportFile;

use Myvetshop\Module\Clinique\Import\Kalivet\Model\AddressModel;
use Myvetshop\Module\Clinique\Import\Kalivet\Model\CartProductModel;
use Myvetshop\Module\Clinique\Import\Kalivet\Model\CustomerModel;
use Myvetshop\Module\Clinique\Import\Kalivet\Model\OrderDetailModel;
use Myvetshop\Module\Clinique\Import\Kalivet\Model\OrderDetailTaxModel;
use Myvetshop\Module\Clinique\Import\Kalivet\Model\OrderModel;
use Myvetshop\Module\Clinique\Import\Kalivet\Model\RewardModel;
use Symfony\Component\Serializer\Serializer;

class DispatchedCustomerData
{
    protected string $dispatchFolder;

    protected string $dispatchedFolder;

    /**
     * @var list<AddressModel>|null
     */
    protected ?array $addresses = null;

    /**
     * @var list<CartProductModel>|null
     */
    protected ?array $cartProducts = null;

    protected ?CustomerModel $customer = null;

    /**
     * @var list<OrderDetailModel>|null
     */
    protected ?array $orderDetails = null;

    /**
     * @var list<OrderDetailTaxModel>|null
     */
    protected ?array $orderDetailTaxes = null;

    /**
     * @var list<OrderModel>|null
     */
    protected ?array $orders = null;

    protected ?RewardModel $reward = null;

    protected Serializer $serializer;

    /**
     * @param string $subfolder
     *
     * @return string[]
     */
    protected function scanSubfolder(string $subfolder): array
    {
        $fullFolder = $this->dispatchFolder
            . $this->dispatchedFolder . DIRECTORY_SEPARATOR
            . $subfolder . DIRECTORY_SEPARATOR;

        if (!\is_dir($fullFolder)) {
            return [];
        }

        return
            \array_values(
                \array_filter(
                    \array_map(
                        function (string $file) use ($fullFolder) {
                            return $fullFolder . $file;
                        },
                        \scandir($fullFolder) ?: [],
                    ),
                    function (string $file) {
                        return \is_file($file);
                    }
                )
            );
    }

    /**
     * @template T
     *
     * @param class-string<T> $className
     * @param list<string> $files
     *
     * @return list<T>
     */
    protected function loadFiles(string $className, array $files): array
    {
        /*
         * @return list<T>
         */
        return \array_map(
            fn ($filename) => $this->serializer->denormalize(
                \json_decode(
                    \file_get_contents($filename) ?: '',
                    true
                ),
                $className,
            ),
            $files
        );
    }

    public function __construct(
        Serializer $serializer,
        string $dispatchFolder,
        string $dispatchedFolder
    ) {
        $this->serializer = $serializer;
        $this->dispatchFolder = $dispatchFolder;
        $this->dispatchedFolder = $dispatchedFolder;
    }

    /**
     * @return list<AddressModel>
     */
    public function getAdresses(): array
    {
        if (null === $this->addresses) {
            $this->addresses = $this->loadFiles(
                AddressModel::class,
                $this->scanSubfolder('ADDRESSES'),
            );
        }

        return $this->addresses;
    }

    /**
     * @return list<CartProductModel>
     */
    public function getCartProducts(): array
    {
        if (null === $this->cartProducts) {
            $this->cartProducts = $this->loadFiles(
                CartProductModel::class,
                $this->scanSubfolder('CART_PRODUCTS'),
            );
        }

        return $this->cartProducts;
    }

    public function getCustomer(): CustomerModel
    {
        if (null === $this->customer) {
            $customers = $this->loadFiles(
                CustomerModel::class,
                $this->scanSubfolder('CUSTOMERS')
            );

            if (\count($customers) !== 1) {
                throw new \Exception(sprintf('Invalid CUSTOMERS data for %s (%s fichiers)', $this->dispatchedFolder, \count($customers)));
            }

            $this->customer = $customers[0];
        }

        return $this->customer;
    }

    /**
     * @return list<OrderDetailModel>
     */
    public function getOrderDetails(): array
    {
        if (null === $this->orderDetails) {
            $this->orderDetails = $this->loadFiles(
                OrderDetailModel::class,
                $this->scanSubfolder('ORDER_DETAILS'),
            );
        }

        return $this->orderDetails;
    }

    /**
     * @return list<OrderDetailTaxModel>
     */
    public function getOrderDetailTaxes(): array
    {
        if (null === $this->orderDetailTaxes) {
            $this->orderDetailTaxes = $this->loadFiles(
                OrderDetailTaxModel::class,
                $this->scanSubfolder('ORDER_DETAIL_TAXES'),
            );
        }

        return $this->orderDetailTaxes;
    }

    /**
     * @return list<OrderModel>
     */
    public function getOrders(): array
    {
        if (null === $this->orders) {
            $this->orders = $this->loadFiles(
                OrderModel::class,
                $this->scanSubfolder('ORDERS'),
            );
        }

        return $this->orders;
    }

    public function getReward(): ?RewardModel
    {
        if (null === $this->reward) {
            $rewards = $this->loadFiles(
                RewardModel::class,
                $this->scanSubfolder('REWARDS'),
            );

            $this->reward = $rewards[0] ?? null;
        }

        return $this->reward;
    }
}
