<?php

namespace Myvetshop\Module\Clinique\Command\Import;

use Myvetshop\Module\Clinique\Import\Kalivet\Clinic\ClinicImporter;
use Myvetshop\Module\Clinique\Import\Kalivet\Clinic\ClinicImportFileProvider;
use Myvetshop\Module\Clinique\Import\Kalivet\ImportFile\DispatchedCustomerDataFactory;
use Myvetshop\Module\Clinique\Import\Kalivet\ImportFile\ImportDispatcher;
use Myvetshop\Module\Clinique\Import\Kalivet\ImportFile\ImportFileParser;
use Myvetshop\Module\Clinique\Import\Kalivet\ImportFile\ImportFileProvider;
use Myvetshop\Module\Clinique\Import\Kalivet\Sync\KalivetDataSyncer;
use Myvetshop\Module\Clinique\Import\Kalivet\Sync\SyncStatistics;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Helper\ProgressBar;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

class KalivetImportCommand extends Command
{
    protected ClinicImportFileProvider $clinicImportFileProvider;
    protected ClinicImporter $clinicImporter;
    protected KalivetDataSyncer $kalivetDataSyncer;
    protected DispatchedCustomerDataFactory $customerDataFactory;
    protected ImportDispatcher $importDispatcher;
    protected ImportFileProvider $importFileProvider;

    /**
     * @return array<string, \Clinique>
     */
    protected function importClinics(SymfonyStyle $output): array
    {
        $ret = [];

        $output->title('Syncinc clinics...');

        $clinics = $this->clinicImportFileProvider->getClinics();
        $progressBar = new ProgressBar($output, \count($clinics));
        foreach ($clinics as $i => $clinic) {
            $progressBar->setProgress($i);
            $ret[$clinic->codeAlcyonKalivet] = $this->clinicImporter->import($clinic);
        }
        $progressBar->finish();

        return $ret;
    }

    public function __construct(
        ClinicImportFileProvider $clinicImportFileProvider,
        ClinicImporter $clinicImporter,
        KalivetDataSyncer $kalivetDataSyncer,
        DispatchedCustomerDataFactory $customerDataFactory,
        ImportDispatcher $importDispatcher,
        ImportFileProvider $importFileProvider
    ) {
        $this->clinicImportFileProvider = $clinicImportFileProvider;
        $this->clinicImporter = $clinicImporter;
        $this->kalivetDataSyncer = $kalivetDataSyncer;
        $this->customerDataFactory = $customerDataFactory;
        $this->importDispatcher = $importDispatcher;
        $this->importFileProvider = $importFileProvider;

        parent::__construct('app:import:kalivet');
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        // Init Context with a new Cart
        $context = \Context::getContext();
        \assert($context instanceof \Context);
        if (!$context->cart) {
            $context->cart = new \Cart();
        }

        // Load module
        \Module::getInstanceByName('myvetshopclinique');

        $styledOutput = new SymfonyStyle($input, $output);

        // Download export
        $styledOutput->title('Downloading export from Kalivet...');
        if ($this->importFileProvider->isCached()) {
            $styledOutput->text('Export cached, no download required');
        } else {
            $this->importFileProvider->download();
        }

        $importFile = $this->importFileProvider->open();

        // Count lines of the import file
        $lineCount = 0;
        while (!\feof($importFile)) {
            // Load 1Mb of file at each loop
            $line = \fgets($importFile, 1048576);
            if (false === $line) {
                break;
            }
            $lineCount = $lineCount + \substr_count($line, "\n");
        }

        \rewind($importFile);
        $styledOutput->text($lineCount . ' lines to process');

        // Pre-process export file
        $styledOutput->title('Pre-processing export file...');
        $parser = new ImportFileParser($importFile);

        $progressBar = new ProgressBar($styledOutput, $lineCount);
        $currentSection = null;
        foreach ($parser as $lineNumber => $value) {
            $newSection = $parser->getSection();
            if ($newSection !== $currentSection) {
                $styledOutput->section($newSection);
                $currentSection = $newSection;
            }

            $progressBar->setProgress($lineNumber);

            if (\is_string($currentSection)) {
                $this->importDispatcher->dispatchLine($currentSection, $value);
            }
        }

        $progressBar->finish();

        // Sync clinics
        $clinicMap = $this->importClinics($styledOutput);

        // Process per-customer import
        $folders = $this->importDispatcher->listFolders();
        $styledOutput->title('Processing customer data');
        $progressBar = new ProgressBar($styledOutput, \count($folders));

        $statistics = new SyncStatistics();
        foreach ($folders as $i => $folder) {
            $progressBar->setProgress($i);

            $customerData = $this->customerDataFactory->create($folder);

            try {
                $this->kalivetDataSyncer->sync($clinicMap, $customerData, $statistics);
            } catch (\Exception $e) {
                $styledOutput->error('Error processing ' . $customerData->getCustomer()->email . ' : ' . $e->getMessage());
                $styledOutput->error($e->getFile() . '#' . $e->getLine());
                $styledOutput->error($e->getTraceAsString());
            }
        }

        $progressBar->finish();

        $styledOutput->text($statistics->nbCustomersCreated . ' customers created');
        $styledOutput->text($statistics->nbCustomersUpdated . ' customers updated');
        $styledOutput->text($statistics->nbAddressesCreated . ' addresses created');
        $styledOutput->text($statistics->nbAddressesUpdated . ' addresses updated');
        $styledOutput->text($statistics->nbOrdersCreated . ' orders created');
        $styledOutput->text($statistics->nbOrdersUpdated . ' orders updated');

        return 0;
    }
}
