<?php

class OrderProblemSomme implements OrderProblemInterface
{
    /**
     * @var Order
     */
    protected $order;

    /**
     * @var OrderDetail[]
     */
    protected $details;

    /**
     * @var OrderCartRule[]
     */
    protected $discounts;

    /**
     * @var string[];
     */
    protected $errors;

    /**
     * @var bool
     */
    protected $repairable;

    /**
     * @return array<string, float>
     */
    protected function getOrderSubtotals()
    {
        return [
            // Montant des produits HT
            'product_total_tax_excl' => round(array_reduce(
                $this->details,
                function ($sum, OrderDetail $orderDetail) {
                    return $sum + $orderDetail->total_price_tax_excl;
                },
                0
            ), 2),
            // Montant des produits TTC
            'product_total_tax_incl' => round(array_reduce(
                $this->details,
                function ($sum, OrderDetail $orderDetail) {
                    return $sum + $orderDetail->total_price_tax_incl;
                },
                0
            ), 2),
            // Montant total des réductions (HT)
            'discount_total_tax_excl' => round(array_reduce(
                $this->discounts,
                function ($sum, OrderCartRule $orderCartRule) {
                    return $sum + $orderCartRule->value_tax_excl;
                },
                0
            ), 2),
            // Montant total des réductions (TTC)
            'discount_total_tax_incl' => round(array_reduce(
                $this->discounts,
                function ($sum, OrderCartRule $orderCartRule) {
                    return $sum + $orderCartRule->value;
                },
                0
            ), 2),
        ];
    }

    /**
     * @param bool $repairable
     *
     * @return string[]
     */
    protected function computeErrors(&$repairable): array
    {
        $ret = [];
        $repairable = true;

        $subTotals = $this->getOrderSubtotals();

        if (round(floatval($this->order->total_products), 2) != $subTotals['product_total_tax_excl']) {
            $ret[] = 'Commande : o.total_products != sum(od.total_price_tax_excl)';
        }

        if (round(floatval($this->order->total_products_wt), 2) != $subTotals['product_total_tax_incl']) {
            $ret[] = 'Commande : o.total_products_wt != sum(od.total_price_tax_incl)';
        }

        if (round(floatval($this->order->total_discounts_tax_excl), 2) != $subTotals['discount_total_tax_excl']) {
            $ret[] = 'Commande : o.total_discounts_tax_excl != sum(ocr.value_tax_excl)';
        }

        if (round(floatval($this->order->total_discounts_tax_incl), 2) != $subTotals['discount_total_tax_incl']) {
            $ret[] = 'Commande : o.total_discounts_tax_incl != sum(ocr.value)';
        }

        // Si le total HT n'est pas bon, erreur
        if (round($subTotals['product_total_tax_excl'] + round(floatval($this->order->total_shipping_tax_excl), 2) - $subTotals['discount_total_tax_excl'], 2) != round(floatval($this->order->total_paid_tax_excl), 2)) {
            $right = round($subTotals['product_total_tax_excl'] + round(floatval($this->order->total_shipping_tax_excl), 2) - $subTotals['discount_total_tax_excl'], 2);
            $left = round(floatval($this->order->total_paid_tax_excl), 2);
            $ret[] = 'Commande : o.total_paid_tax_excl != sum(od.product_total_tax_excl) + o.total_shipping_tax_excl - SUM(ocr.value_tax_excl) [' . $left . ' != ' . $right . ']';
        }

        // Si le total TTC n'et pas bon, erreur
        if (round($subTotals['product_total_tax_incl'] + round(floatval($this->order->total_shipping_tax_incl), 2) - $subTotals['discount_total_tax_incl'], 2) != round(floatval($this->order->total_paid_tax_incl), 2)) {
            $right = round($subTotals['product_total_tax_incl'] + round(floatval($this->order->total_shipping_tax_incl), 2) - $subTotals['discount_total_tax_incl'], 2);
            $left = round(floatval($this->order->total_paid_tax_incl), 2);

            $ret[] = 'Commande : o.discount_total_tax_incl != sum(od.product_total_tax_incl) + o.total_shipping_tax_incl - SUM(ocr.value) [' . $left . ' != ' . $right . ']';
            $repairable = false;
        }

        return $ret;
    }

    public function __construct(Order $order)
    {
        $this->order = $order;
        $this->details = ObjectModel::hydrateCollection(OrderDetail::class, $order->getOrderDetailList());
        $this->discounts = ObjectModel::hydrateCollection(OrderCartRule::class, $order->getCartRules());
        $this->errors = $this->computeErrors($this->repairable);
    }

    public function getErrorMessage(): string
    {
        return 'Commande : Les sommes des produits et fdp ne correspondent pas au total de la commande';
    }

    /**
     * @return string[]
     */
    public function getErrors(): array
    {
        return $this->errors;
    }

    public function isProblematic(): bool
    {
        return count($this->errors) > 0;
    }

    public function isRepairable(): bool
    {
        return $this->isProblematic() && $this->repairable;
    }

    public function repair(): bool
    {
        $subTotals = $this->getOrderSubtotals();

        $this->order->total_products = $subTotals['product_total_tax_excl'];
        $this->order->total_products_wt = $subTotals['product_total_tax_incl'];
        $this->order->total_discounts_tax_excl = $subTotals['discount_total_tax_excl'];
        $this->order->total_discounts_tax_incl = $subTotals['discount_total_tax_incl'];

        // Si le total HT n'est pas bon, erreur
        $this->order->total_paid_tax_excl = round($subTotals['product_total_tax_excl'] + $this->order->total_shipping_tax_excl - $subTotals['discount_total_tax_excl'], 2);

        return true;
    }
}
