<?php
declare(strict_types=1);
/**
 * Created by Aurélien RICHAUD (01/08/2019 12:16)
 */
require_once __DIR__ . DIRECTORY_SEPARATOR . 'ApiFrontController.php';
require_once __DIR__ . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . 'classes' . DIRECTORY_SEPARATOR . 'Clinique.php';

class MyvetshopcliniqueOrdersModuleFrontController extends ApiFrontController
{
    /**
     * @return array<string, mixed>
     */
    protected function serializeOrder(Order $order): array
    {
        $orderState = new OrderState($order->current_state);
        $orderCarrier = Carrier::getCarrierByReference($order->id_carrier);

        $ret = [
            'id' => $order->id,
            'id_address_delivery' => $order->id_address_delivery,
            'id_address_invoice' => $order->id_address_invoice,
            'id_cart' => $order->id_cart,
            'id_currency' => $order->id_currency,
            'id_lang' => $order->id_lang,
            'id_customer' => $order->id_customer,
            'id_carrier' => $order->id_carrier,
            'carrier' => [
                'id' => $orderCarrier->id,
                'name' => $orderCarrier->name,
                'is_free' => boolval($orderCarrier->is_free),
                'url' => $orderCarrier->url,
                'delay' => $orderCarrier->delay,
                'domicile' => boolval($orderCarrier->domicile),
            ],
            'current_state' => $order->current_state,
            'module' => $order->module,
            'invoice_number' => $order->invoice_number,
            'invoice_date' => $order->invoice_date,
            'delivery_number' => $order->delivery_number,
            'delivery_date' => $order->delivery_date,
            'valid' => $order->valid,
            'date_add' => $order->date_add,
            'date_upd' => $order->date_upd,
            'shipping_number' => $order->shipping_number,
            'id_shop_group' => $order->id_shop_group,
            'id_shop' => $order->id_shop,
            'secure_key' => $order->secure_key,
            'payment' => $order->payment,
            'recyclable' => $order->recyclable,
            'gift' => $order->gift,
            'gift_message' => $order->gift_message,
            'mobile_theme' => $order->mobile_theme,
            'total_discounts' => $order->total_discounts,
            'total_discounts_tax_incl' => $order->total_discounts_tax_incl,
            'total_discounts_tax_excl' => $order->total_discounts_tax_excl,
            'total_paid' => $order->total_paid,
            'total_paid_tax_incl' => $order->total_paid_tax_incl,
            'total_paid_tax_excl' => $order->total_paid_tax_excl,
            'total_paid_real' => $order->total_paid_real,
            'total_products' => $order->total_products,
            'total_products_wt' => $order->total_products_wt,
            'total_shipping' => $order->total_shipping,
            'total_shipping_tax_incl' => $order->total_shipping_tax_incl,
            'total_shipping_tax_excl' => $order->total_shipping_tax_excl,
            'carrier_tax_rate' => $order->carrier_tax_rate,
            'total_wrapping' => $order->total_wrapping,
            'total_wrapping_tax_incl' => $order->total_wrapping_tax_incl,
            'total_wrapping_tax_excl' => $order->total_wrapping_tax_excl,
            'round_mode' => $order->round_mode,
            'round_type' => $order->round_type,
            'conversion_rate' => $order->conversion_rate,
            'reference' => $order->reference,
            'products' => [],
            'order_state' => [
                'id' => $orderState->id,
                'unremovable' => $orderState->unremovable,
                'delivery' => $orderState->delivery,
                'hidden' => $orderState->hidden,
                'send_email' => $orderState->send_email,
                'module_name' => $orderState->module_name,
                'invoice' => $orderState->invoice,
                'color' => $orderState->color,
                'logable' => $orderState->logable,
                'shipped' => $orderState->shipped,
                'paid' => $orderState->paid,
                'pdf_delivery' => $orderState->pdf_delivery,
                'pdf_invoice' => $orderState->pdf_invoice,
                'deleted' => $orderState->deleted,
                'name' => $orderState->name[1],
                'template' => $orderState->template,
            ],
            'history' => [],
        ];

        foreach ($order->getProductsDetail() as $productDetail) {
            $product = [
                'id' => $productDetail['id_order_detail'],
                'product_id' => $productDetail['product_id'],
                'product_attribute_id' => $productDetail['product_attribute_id'],
                'product_name' => $productDetail['product_name'],
                'product_reference' => $productDetail['product_reference'],
                'product_ean13' => $productDetail['product_ean13'],
                'product_upc' => $productDetail['product_upc'],
                'unit_price_tax_excl' => $productDetail['unit_price_tax_excl'],
                'unit_price_tax_incl' => $productDetail['unit_price_tax_incl'],
                'product_quantity' => $productDetail['product_quantity'],
                'total_price_tax_incl' => $productDetail['total_price_tax_incl'],
                'total_price_tax_excl' => $productDetail['total_price_tax_excl'],
            ];

            $ret['products'][] = $product;
        }

        foreach ($order->getHistory($this->context->language->id) as $orderHistory) {
            $ret['history'][] = $orderHistory;
        }

        return $ret;
    }

    /**
     * @return array<string, mixed>
     *
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    protected function actionLastOrder(): array
    {
        $customer = $this->context->customer;

        $lastOrderId = Db::getInstance(false)
            ->getValue('SELECT `id_order` FROM `' . _DB_PREFIX_ . 'orders` WHERE `id_customer` = ' . (int) $customer->id . ' ORDER BY `id_order` DESC');

        if (!$lastOrderId) {
            header('HTTP/1.0 500 Internal Server Error');

            return [
                'message' => 'No order',
            ];
        }

        $order = new Order($lastOrderId);

        return $this->serializeOrder($order);
    }

    /**
     * @return array<int, array<string, mixed>>
     *
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    protected function actionLastOrders(): array
    {
        $customer = $this->context->customer;

        $lastOrdersId = Db::getInstance(false)
            ->executeS('SELECT `id_order` FROM `' . _DB_PREFIX_ . 'orders` WHERE `id_customer` = ' . (int) $customer->id . ' ORDER BY `id_order` DESC LIMIT 10');

        if (!is_array($lastOrdersId)) {
            $lastOrdersId = [];
        }

        $ret = [];
        foreach ($lastOrdersId as $row) {
            $order = new Order($row['id_order']);
            $ret[] = $this->serializeOrder($order);
        }

        return $ret;
    }

    /**
     * @return array<string, mixed>
     *
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    protected function actionOrder(): array
    {
        $customer = $this->context->customer;

        $orderIdFiltered = Db::getInstance(false)
            ->getValue(
                'SELECT `id_order` FROM `' . _DB_PREFIX_ . 'orders` WHERE `id_customer` = ' . (int) $customer->id . ' AND `id_order` = ' . (int) $_GET['order_id']
            );

        if (!$orderIdFiltered) {
            return [];
        }

        $order = new Order($orderIdFiltered);

        return $this->serializeOrder($order);
    }

    /**
     * @return string[]
     *
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    protected function actionOrderPaid(): array
    {
        $idCart = (int) Tools::getValue('id_cart', null);
        $idOrder = (int) Tools::getValue('id_order', null);
        $idModule = (int) (Tools::getValue('id_module', 0));
        $key = Tools::getValue('key', null);

        $cart = new Cart($idCart);

        if ($cart->id_customer != $this->context->customer->id || $cart->secure_key != $key) {
            header('HTTP/1.0 403 Forbidden');

            return ['error' => 'id_customer,secure_key'];
        }

        $ret = [
            'status' => Tools::getValue('status'),
            'id_order' => (int) $idOrder,
            'id_cart' => (int) $idCart,
        ];

        if (!$idOrder) {
            /* @phpstan-ignore-next-line Erreur de typage dans le code source prestashop (retorune l'id de la commande) */
            $idOrder = (int) Order::getOrderByCartId((int) $idCart);
        }

        if ($idOrder) {
            $order = new Order($idOrder);
            $module = Module::getInstanceById((int) $idModule);
            if ($order->module != $module->name) {
                header('HTTP/1.0 403 Forbidden');

                return ['error' => 'module_name'];
            }

            $orderStatus = $order->getCurrentOrderState();
            if ($orderStatus->paid) {
                $ret['status'] = 'success';
            } else {
                $ret['status'] = 'reject';
            }

            // Ajout d'une note dans la commande pour indiquer qu'elle vient de l'application mobile
            $msg = new Message();
            $message = strip_tags("Cette commande a été passée depuis l'application mobile", '<br>');
            if (Validate::isCleanHtml($message)) {
                $msg->message = $message;
                $msg->id_cart = (int) $idCart;
                $msg->id_customer = (int) ($order->id_customer);
                $msg->id_order = (int) $order->id;
                $msg->private = true;
                try {
                    $msg->add();
                } catch (PrestaShopDatabaseException $e) {
                    // Inhibiteur d'erreur
                } catch (PrestaShopException $e) {
                    // Inhibiteur d'erreur
                }
            }
        }

        header('Content-Type: text/html'); ?>
        <!doctype html>
        <html lang="en">
        <head>
            <meta charset="utf-8">
            <script type="text/javascript">
                window.orderPaid = <?php echo json_encode($ret); ?>;
            </script>
        </head>
        <body style="color: white; background-color: white;">
        Fin du processus de commande
        </body>
        </html>
        <?php
        exit();
    }

    /**
     * @return array<string, mixed>
     */
    protected function actionPayOrder(): array
    {
        $cart = $this->context->cart;

        if (0 == count($cart->getProducts())) {
            return ['success' => false, 'form' => '', 'message' => "Impossible de passer commande d'un panier vide."];
        }

        /** @var MyvetshopPayment $paymentModule */
        $paymentModule = Module::getInstanceByName('myvetshoppayment');

        try {
            header('Content-Type: text/html'); ?>
            <!doctype html>
            <html lang="en">
            <head>
                <meta charset="utf-8">
                <script type="text/javascript">
                    window.onload = function () {
                        document.forms[0].submit();
                    }
                </script>
            </head>
            <body style="color: white; background-color: white;">
            <?php
            echo $paymentModule->hookPayment(
                [
                    'cart' => $cart,
                    'url_retour_ok' => Tools::getHttpHost(true, true, true) . _MODULE_DIR_ . 'myvetshoppayment/submit.php?id_cart=' . intval($cart->id) . '&app=1',
                    'url_retour_err' => Tools::getHttpHost(true, true, true) . _MODULE_DIR_ . 'myvetshoppayment/submit.php?id_cart=' . intval($cart->id) . '&app=1',
                ]
            ); ?>
            </body>
            </html>
            <?php
            exit();
        } catch (PrestaShopException $e) {
            return [
                'success' => false,
                'form' => '',
                'message' => $e->getMessage(),
            ];
        }
    }

    protected function actionPdfInvoice()
    {
        $idOrder = Tools::getValue('id_order', null);
        if (!$idOrder) {
            return ['success' => false, 'message' => 'Aucune commande sélectionnée'];
        }

        $order = new Order($idOrder);
        if ($order->id_customer != $this->context->customer->id) {
            return ['success' => false, 'message' => "Problème d'accès à la commande"];
        }

        Tools::redirectLink('/index.php?controller=pdf-invoice&id_order=' . Tools::getValue('id_order', null) . '&secure_key=' . $order->secure_key);
    }

    /**
     * @return array<string, mixed>
     *
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    protected function actionReorder(): array
    {
        $idOrder = Tools::getValue('id_order', null);

        if (!$idOrder) {
            return ['success' => false, 'message' => 'Aucune commande sélectionnée'];
        }

        $order = new Order($idOrder);
        if ($order->id_customer != $this->context->customer->id) {
            return ['success' => false, 'message' => "Problème d'accès à la commande"];
        }
        $originalCart = new Cart($order->id_cart);

        $duplication = $originalCart->duplicate();
        if (!$duplication || !Validate::isLoadedObject($duplication['cart'])) {
            return ['success' => false, 'message' => 'Impossible de repasser commande'];
        } elseif (!$duplication['success']) {
            return ['success' => false, 'message' => 'Certains produits ne sont plus disponibles'];
        } else {
            $this->context->cookie->id_cart = $duplication['cart']->id;
            $context = $this->context;
            $this->context->cart = $duplication['cart'];
            CartRule::autoAddToCart($context);
            $this->context->cookie->write();

            return ['success' => true, 'message' => 'Commande re-passée', 'id_cart' => $duplication['cart']->id];
        }
    }

    /**
     * Assign template vars related to page content
     *
     * @see FrontController::initContent()
     */
    public function initContent(): void
    {
        parent::initContent();

        switch (Tools::getValue('action', 'nothing')) {
            case 'lastOrder':
                $ret = $this->actionLastOrder();
                break;

            case 'lastOrders':
                $ret = $this->actionLastOrders();
                break;

            case 'order':
                $ret = $this->actionOrder();
                break;

            case 'orderPaid':
                $ret = $this->actionOrderPaid();
                break;

            case 'payOrder':
                $ret = $this->actionPayOrder();
                break;

            case 'pdfInvoice':
                $ret = $this->actionPdfInvoice();
                break;

            case 'reOrder':
                $ret = $this->actionReorder();
                break;

            default:
                $ret = [];
        }

        echo json_encode($ret, JSON_PRETTY_PRINT);
        exit();
    }
}
