<?php

declare(strict_types=1);

require_once __DIR__ . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . 'classes' . DIRECTORY_SEPARATOR . 'Clinique.php';
require_once __DIR__ . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . 'classes' . DIRECTORY_SEPARATOR . 'CliniqueHolidays.php';

/**
 * Created by Aurélien RICHAUD (10/07/2017 16:15)
 */
class AdminMyVetShopCliniqueController extends ModuleAdminController
{
    public const GMAP_API_KEY = 'AIzaSyCass66EL1weNGYcbEBjxHnaUXzXxT642o';

    /**
     * @var Clinique
     */
    protected $object;

    /**
     * @throws PrestaShopException
     */
    protected function allowAccessoires(int $group_id): void
    {
        $db = Db::getInstance();

        // Catégorie d'accessoires
        $catAccessoires = $this->object->getAccessoireCategories();

        if ($catAccessoires) {
            foreach ($catAccessoires as $idCat) {
                $count = $db->getValue('SELECT COUNT(*) FROM `' . _DB_PREFIX_ . 'category_group` WHERE `id_category` = ' . (int) $idCat . ' AND id_group = ' . (int) $group_id);

                if (!$count) {
                    $db->execute('INSERT INTO `' . _DB_PREFIX_ . 'category_group` (`id_category`, `id_group`) VALUES (' . (int) $idCat . ', ' . (int) $group_id . ')');
                }
            }

            // Récupère le ID des catégories d'accessoires à retirer
            $otherCats = array_diff([274, 399, 418], $catAccessoires);
            // Supprime les accès aux catégories d'accessoires des autres centrales
            $db->execute('DELETE FROM `' . _DB_PREFIX_ . 'category_group` WHERE `id_group` = ' . (int) $group_id . ' AND id_category IN (' . implode(', ', $otherCats) . ')');

            // Détermine la catégorie accessoire principale, et applique 10% de réduction
            $mainAccessoireCatId = array_values(array_intersect($catAccessoires, [274, 399, 418]));

            if (count($mainAccessoireCatId)) {
                if (!GroupReduction::doesExist($this->object->id_group, $mainAccessoireCatId[0])) {
                    $groupeReduction = new GroupReduction();
                    $groupeReduction->id_group = $group_id;
                    $groupeReduction->id_category = $mainAccessoireCatId[0];
                    $groupeReduction->reduction = 0.100;
                    $groupeReduction->save();
                }
            }
        }
    }

    public function __construct()
    {
        $this->bootstrap = true;
        $this->context = Context::getContext();
        $this->table = Clinique::TABLE;
        $this->className = Clinique::class;
        $this->allow_export = true;
        $this->lang = false;
        $this->fields_list = [
            'id_myvetshop_clinique' => [
                'title' => $this->l('ID'),
                'align' => 'text-center',
                'class' => 'fixed-width-xs',
            ],
            'id_employee' => [
                'title' => $this->l('Nom'),
                'havingFilter' => false,
                'callback' => 'printEmployee',
            ],
            'id_store' => [
                'title' => $this->l('Adresse'),
                'havingFilter' => false,
                'callback' => 'printAdresse',
            ],
            'code_privilege' => [
                'title' => $this->l('Code privilège'),
                'align' => 'text-center',
                'class' => 'fixed-width-xs',
            ],
            'deleted' => [
                'title' => $this->l('Active'),
                'align' => 'text-center',
                'class' => 'fixed-width-xs',
                'callback' => 'printActive',
            ],
            'centrale' => [
                'title' => $this->l('Centrale'),
                'align' => 'text-center',
                'class' => 'fixed-width-xs',
            ],
            'id_warehouse' => [
                'title' => $this->l('Entrepot'),
                'havingFilter' => false,
                'callback' => 'printEntrepot',
            ],
        ];
        $this->bulk_actions = [
            'delete' => [
                'text' => $this->l('Delete selected'),
                'icon' => 'icon-trash',
                'confirm' => $this->l('Delete selected object?'),
            ],
        ];
        $this->fieldImageSettings = [
            [
                'name' => 'logo',
                'dir' => 'e',
                'id_field' => 'id_employee',
                'height' => 150,
                'width' => 380,
            ],
            [
                'name' => 'photo',
                'dir' => 'st',
                'id_field' => 'id_store',
                'height' => 150,
                'width' => 150,
            ],
        ];

        // Actions disponibles
        $this->addRowAction('view');
        $this->addRowAction('edit');
        //$this->addRowAction('delete');

        parent::__construct();
    }

    /**
     * @throws PrestaShopDatabaseException
     */
    public function initProcess(): void
    {
        if (5 == Context::getContext()->employee->id_profile) {
            // Empêche l'édition
            if (isset($_GET['update' . $this->table])) {
                Tools::redirectAdmin(Context::getContext()->link->getAdminLink('AdminMyVetShopClinique'));
            }

            // Empêche l'export
            if (isset($_GET['export' . $this->table])) {
                Tools::redirectAdmin(Context::getContext()->link->getAdminLink('AdminMyVetShopClinique'));
            }

            // Empêche la suppression
            if (isset($_GET['delete' . $this->table])) {
                Tools::redirectAdmin(Context::getContext()->link->getAdminLink('AdminMyVetShopClinique'));
            }

            $_GET['view' . $this->table] = '';
            $_GET['id_myvetshop_clinique'] = Clinique::getCliniqueByEmployee(Context::getContext()->employee->id);
        }

        parent::initProcess();
    }

    /**
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    public function printEmployee(int $id_employee): string
    {
        $employee = new Employee($id_employee);

        return $employee->firstname . ' ' . $employee->lastname;
    }

    /**
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    public function printAdresse(int $id_store): string
    {
        $store = new Store($id_store);

        $ret = $store->address1 . '<br>';
        $ret .= $store->address2 ? $store->address2 . '<br>' : '';

        return $ret . $store->postcode . ' ' . $store->city;
    }

    public function printActive(bool $deleted): string
    {
        if ($deleted) {
            return '<a class="list-action-enable action-disabled" href="#" title=""><i class="icon-remove"></i></a>';
        } else {
            return '<a class="list-action-enable action-enabled" href="#" title=""><i class="icon-check"></i></a>';
        }
    }

    /**
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    public function printEntrepot(int $id_warehouse): string
    {
        $warehouse = new Warehouse($id_warehouse);

        return $warehouse->name;
    }

    /**
     * @param ObjectModel $obj
     *
     * @return array<string, mixed>|false
     *
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     *
     * {@inheritDoc}
     */
    public function getFieldsValue($obj)
    {
        $ret = parent::getFieldsValue($obj);

        if ($obj instanceof Clinique) {
            $employee = new Employee($obj->id_employee);
            $ret['firstname'] = $employee->firstname;
            $ret['lastname'] = $employee->lastname;

            $store = new Store($obj->id_store);
            $ret['address1'] = $store->address1;
            $ret['address2'] = $store->address2;
            $ret['postcode'] = $store->postcode;
            $ret['city'] = $store->city;
            $ret['latitude'] = $store->latitude;
            $ret['longitude'] = $store->longitude;
            $ret['website'] = $store->website;  /* @phpstan-ignore-line */
        }

        return $ret;
    }

    /**
     * @throws Exception
     * @throws SmartyException
     */
    public function renderForm(): string
    {
        /////////////////////////////////////////
        /// Détection des photos
        $logo_url = '';
        $logo_size = 0;
        $photo_url = '';
        $photo_size = 0;
        /** @var Clinique|false $obj */
        $obj = $this->loadObject(true);
        if ($obj) {
            // Logo
            $logo = _PS_EMPLOYEE_IMG_DIR_ . $obj->id_employee . '.jpg';
            $logo_url = ImageManager::thumbnail($logo, 'employee_' . (int) $obj->id_employee . '.' . $this->imageType, 150, $this->imageType, true, true);
            $logo_size = file_exists($logo) ? filesize($logo) / 1000 : false;

            // Photo de la clinique
            $photo = _PS_STORE_IMG_DIR_ . $obj->id_store . '.jpg';
            $photo_url = ImageManager::thumbnail($photo, 'store_' . (int) $obj->id_store . '.' . $this->imageType, 300, $this->imageType, true, true);
            $photo_size = file_exists($photo) ? filesize($photo) / 1000 : false;
        }

        $this->multiple_fieldsets = true;
        $this->fields_form = [
            [
                'form' => [
                    'legend' => [
                        'title' => (Tools::getValue('id_myvetshop_clinique')) ? $this->l('Modifier une clinique') : $this->l('Ajouter une clinique'),
                        'icon' => 'icon-angle-double-right',
                    ],
                    'input' => [
                        [
                            'type' => 'text',
                            'label' => $this->l('Code privilège'),
                            'name' => 'code_privilege',
                            'required' => true,
                            'col' => '1',
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Prénom'),
                            'name' => 'firstname',
                            'required' => false,
                            'col' => 2,
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Nom'),
                            'name' => 'lastname',
                            'required' => false,
                            'col' => 2,
                        ],
                        [
                            'type' => 'select',
                            'label' => $this->l('Thème'),
                            'name' => 'theme',
                            'empty_message' => '',
                            'options' => [
                                'query' => [
                                    ['id' => '', 'label' => 'Aucun'],
                                    ['id' => 'theme1', 'label' => 'Bleu'],
                                    ['id' => 'theme2', 'label' => 'Sable'],
                                    ['id' => 'theme3', 'label' => 'Orange / Jaune'],
                                    ['id' => 'theme4', 'label' => 'Turquoise / Vert'],
                                    ['id' => 'theme5', 'label' => 'Bleu / Jaune'],
                                    ['id' => 'theme6', 'label' => 'Bleu / Jaune / Vert'],
                                    ['id' => 'theme7', 'label' => 'Violet'],
                                    ['id' => 'theme8', 'label' => 'Bleu / Rose / Vert'],
                                    ['id' => 'theme9', 'label' => 'Noir / Bleu / Vert'],
                                    ['id' => 'theme10', 'label' => 'Vert / Noir'],
                                    ['id' => 'theme11', 'label' => 'Noir / Rouge / Vert'],
                                ],
                                'id' => 'id',
                                'name' => 'label',
                            ],
                            'required' => false,
                        ],
                        [
                            'type' => 'file',
                            'label' => $this->l('Logo (150x380)'),
                            'name' => 'logo',
                            'required' => false,
                            'display_image' => true,
                            'image' => $logo_url ? $logo_url : false,
                            'size' => $logo_size,
                        ],
                        [
                            'type' => 'file',
                            'label' => $this->l('Photo de la clinique (150x150)'),
                            'name' => 'photo',
                            'required' => false,
                            'display_image' => true,
                            'image' => $photo_url ? $photo_url : false,
                            'size' => $photo_size,
                        ],
                        [
                            'type' => 'radio',
                            'label' => $this->l('Affichage du logo'),
                            'name' => 'masquer_logo',
                            'empty_message' => '',
                            'values' => [
                                [
                                    'id' => 'afficher_myvetshop',
                                    'value' => 0,
                                    'label' => 'Afficher le logo MyVetshop',
                                ],
                                [
                                    'id' => 'masquer_myvetshop',
                                    'value' => 1,
                                    'label' => 'Masquer le logo MyVetshop',
                                ],
                            ],
                            'required' => false,
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('E-Mail pour les ARCs'),
                            'name' => 'email_original',
                            'required' => false,
                            'col' => 2,
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Site internet'),
                            'name' => 'website',
                            'required' => false,
                            'col' => 2,
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Adresse 1'),
                            'name' => 'address1',
                            'required' => false,
                            'col' => 2,
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Adresse 2'),
                            'name' => 'address2',
                            'required' => false,
                            'col' => 2,
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Code postal'),
                            'name' => 'postcode',
                            'required' => false,
                            'col' => 1,
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Ville'),
                            'name' => 'city',
                            'required' => false,
                            'col' => 2,
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Latitude'),
                            'name' => 'latitude',
                            'required' => false,
                            'col' => 2,
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Longitude'),
                            'name' => 'longitude',
                            'required' => false,
                            'col' => 2,
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('URL de prise de rendez-vous'),
                            'name' => 'url_rendez_vous',
                            'required' => false,
                            'col' => 4,
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Horaire Lundi'),
                            'name' => 'hours_lundi',
                            'required' => false,
                            'col' => 4,
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Horaire Mardi'),
                            'name' => 'hours_mardi',
                            'required' => false,
                            'col' => 4,
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Horaire Mercredi'),
                            'name' => 'hours_mercredi',
                            'required' => false,
                            'col' => 4,
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Horaire Jeudi'),
                            'name' => 'hours_jeudi',
                            'required' => false,
                            'col' => 4,
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Horaire Vendredi'),
                            'name' => 'hours_vendredi',
                            'required' => false,
                            'col' => 4,
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Horaire Samedi'),
                            'name' => 'hours_samedi',
                            'required' => false,
                            'col' => 4,
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Horaire Dimanche'),
                            'name' => 'hours_dimanche',
                            'required' => false,
                            'col' => 4,
                        ],
                        [
                            'type' => 'checkbox',
                            'label' => $this->l('Jours de livraison'),
                            'name' => 'jours_livraison',
                            'required' => false,
                            'values' => [
                                'query' => [
                                    [
                                        'id' => 'lundi',
                                        'name' => $this->l('Lundi'),
                                        'val' => 'lundi',
                                    ],
                                    [
                                        'id' => 'mardi',
                                        'name' => $this->l('Mardi'),
                                        'val' => 'mardi',
                                    ],
                                    [
                                        'id' => 'mercredi',
                                        'name' => $this->l('Mercredi'),
                                        'val' => 'mercredi',
                                    ],
                                    [
                                        'id' => 'jeudi',
                                        'name' => $this->l('Jeudi'),
                                        'val' => 'jeudi',
                                    ],
                                    [
                                        'id' => 'vendredi',
                                        'name' => $this->l('Vendredi'),
                                        'val' => 'vendredi',
                                    ],
                                    [
                                        'id' => 'samedi',
                                        'name' => $this->l('Samedi'),
                                        'val' => 'samedi',
                                    ],
                                    [
                                        'id' => 'dimanche',
                                        'name' => $this->l('Dimanche'),
                                        'val' => 'dimanche',
                                    ],
                                ],
                                'id' => 'id',
                                'name' => 'name',
                            ],
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Tarif livraison à domicile (H.T)'),
                            'name' => 'carrier_home_price',
                            'required' => false,
                            'col' => 1,
                        ],
                        [
                            'type' => 'radio',
                            'label' => $this->l('Rurale'),
                            'name' => 'rurale',
                            'empty_message' => '',
                            'values' => [
                                [
                                    'id' => 'desactiver_rurale',
                                    'value' => 0,
                                    'label' => 'Masquer la rurale pour cette clinique',
                                ],
                                [
                                    'id' => 'activer_rurale',
                                    'value' => 1,
                                    'label' => 'Activer la rurale pour cette clinique',
                                ],
                            ],
                            'required' => false,
                        ],
                        [
                            'type' => 'textarea',
                            'label' => $this->l('Message avant la commande'),
                            'name' => 'message',
                            'required' => false,
                            'cols' => 10,
                            'rows' => 10,
                            'class' => 'rte autoload_rte',
                        ],
                        [
                            'type' => 'radio',
                            'label' => $this->l('Don Univet Nature'),
                            'name' => 'don',
                            'empty_message' => '',
                            'values' => [
                                [
                                    'id' => 'desactiver_don',
                                    'value' => 0,
                                    'label' => 'Masquer le don pour cette clinique',
                                ],
                                [
                                    'id' => 'activer_don',
                                    'value' => 1,
                                    'label' => 'Activer le don pour cette clinique',
                                ],
                            ],
                            'required' => true,
                        ],
                    ],
                ],
            ],
            [
                'form' => [
                    'legend' => [
                        'title' => 'Vétérinaire 1',
                        'icon' => 'icon-user-md',
                    ],
                    'input' => [
                        [
                            'type' => 'text',
                            'label' => $this->l('Prénom'),
                            'name' => 'vetoAprenom',
                            'required' => false,
                            'col' => '2',
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Nom'),
                            'name' => 'vetoAnom',
                            'required' => false,
                            'col' => '2',
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('URL Téléconsultation'),
                            'name' => 'vetoAurlrdv',
                            'required' => false,
                            'col' => '6',
                        ],
                    ],
                ],
            ],
            [
                'form' => [
                    'legend' => [
                        'title' => 'Vétérinaire 2',
                        'icon' => 'icon-user-md',
                    ],
                    'input' => [
                        [
                            'type' => 'text',
                            'label' => $this->l('Prénom'),
                            'name' => 'vetoBprenom',
                            'required' => false,
                            'col' => '2',
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Nom'),
                            'name' => 'vetoBnom',
                            'required' => false,
                            'col' => '2',
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('URL Téléconsultation'),
                            'name' => 'vetoBurlrdv',
                            'required' => false,
                            'col' => '6',
                        ],
                    ],
                ],
            ],
            [
                'form' => [
                    'legend' => [
                        'title' => 'Vétérinaire 3',
                        'icon' => 'icon-user-md',
                    ],
                    'input' => [
                        [
                            'type' => 'text',
                            'label' => $this->l('Prénom'),
                            'name' => 'vetoCprenom',
                            'required' => false,
                            'col' => '2',
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Nom'),
                            'name' => 'vetoCnom',
                            'required' => false,
                            'col' => '2',
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('URL Téléconsultation'),
                            'name' => 'vetoCurlrdv',
                            'required' => false,
                            'col' => '6',
                        ],
                    ],
                ],
            ],
            [
                'form' => [
                    'legend' => [
                        'title' => 'Vétérinaire 4',
                        'icon' => 'icon-user-md',
                    ],
                    'input' => [
                        [
                            'type' => 'text',
                            'label' => $this->l('Prénom'),
                            'name' => 'vetoDprenom',
                            'required' => false,
                            'col' => '2',
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Nom'),
                            'name' => 'vetoDnom',
                            'required' => false,
                            'col' => '2',
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('URL Téléconsultation'),
                            'name' => 'vetoDurlrdv',
                            'required' => false,
                            'col' => '6',
                        ],
                    ],
                ],
            ],
            [
                'form' => [
                    'legend' => [
                        'title' => 'Vétérinaire 5',
                        'icon' => 'icon-user-md',
                    ],
                    'input' => [
                        [
                            'type' => 'text',
                            'label' => $this->l('Prénom'),
                            'name' => 'vetoEprenom',
                            'required' => false,
                            'col' => '2',
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Nom'),
                            'name' => 'vetoEnom',
                            'required' => false,
                            'col' => '2',
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('URL Téléconsultation'),
                            'name' => 'vetoEurlrdv',
                            'required' => false,
                            'col' => '6',
                        ],
                    ],
                ],
            ],
            [
                'form' => [
                    'legend' => [
                        'title' => 'Centrale',
                        'icon' => 'icon-angle-double-right',
                    ],
                    'input' => [
                        [
                            'type' => 'select',
                            'label' => $this->l('Centrale'),
                            'name' => 'centrale',
                            'required' => false,
                            'col' => '1',
                            'options' => [
                                'query' => [
                                    ['id' => '', 'label' => ''],
                                    ['id' => 'alcyon', 'label' => 'Alcyon'],
                                    ['id' => 'centravet', 'label' => 'CentraVet'],
                                    ['id' => 'coveto', 'label' => 'CoVeto'],
                                    ['id' => 'hippocampe', 'label' => 'Hippocampe'],
                                ],
                                'id' => 'id',
                                'name' => 'label',
                            ],
                        ],
                        [
                            'type' => 'select',
                            'label' => $this->l('Entrepot'),
                            'name' => 'id_warehouse',
                            'required' => false,
                            'col' => '1',
                            'options' => [
                                'query' => Db::getInstance()->executeS('SELECT `id_warehouse`, `name` FROM `' . _DB_PREFIX_ . 'warehouse`'),
                                'id' => 'id_warehouse',
                                'name' => 'name',
                            ],
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Code client'),
                            'name' => 'code_client',
                            'required' => false,
                            'col' => '1',
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Mot de passe'),
                            'name' => 'mdp',
                            'required' => false,
                            'col' => '1',
                        ],
                        [
                            'type' => 'text',
                            'label' => $this->l('Code filière'),
                            'name' => 'code_filiere',
                            'required' => false,
                            'col' => '1',
                        ],
                    ],
                    'submit' => [
                        'title' => $this->l('Save'),
                        'class' => 'btn btn-default pull-right',
                        'name' => 'saveAction',
                    ],
                ],
            ],
        ];

        $hours = explode(';', (new Store($this->object->id_store))->getWsHours());
        $jours_livraison = explode(',', $this->object->jours_livraison);
        $carrier_home_price_default = 14.0;

        $carrier_home = new Carrier($this->object->id_carrier_home);
        $price = $carrier_home->getDeliveryPriceByWeight(50, 1);

        $this->fields_value = [
            'hours_lundi' => isset($hours[0]) ? $hours[0] : '',
            'hours_mardi' => isset($hours[1]) ? $hours[1] : '',
            'hours_mercredi' => isset($hours[2]) ? $hours[2] : '',
            'hours_jeudi' => isset($hours[3]) ? $hours[3] : '',
            'hours_vendredi' => isset($hours[4]) ? $hours[4] : '',
            'hours_samedi' => isset($hours[5]) ? $hours[5] : '',
            'hours_dimanche' => isset($hours[6]) ? $hours[6] : '',
            'jours_livraison_lundi' => in_array('lundi', $jours_livraison) ? true : false,
            'jours_livraison_mardi' => in_array('mardi', $jours_livraison) ? true : false,
            'jours_livraison_mercredi' => in_array('mercredi', $jours_livraison) ? true : false,
            'jours_livraison_jeudi' => in_array('jeudi', $jours_livraison) ? true : false,
            'jours_livraison_vendredi' => in_array('vendredi', $jours_livraison) ? true : false,
            'jours_livraison_samedi' => in_array('samedi', $jours_livraison) ? true : false,
            'jours_livraison_dimanche' => in_array('dimanche', $jours_livraison) ? true : false,
            'carrier_home_price' => round($price ? $price : $carrier_home_price_default, 2),
        ];

        return parent::renderForm();
    }

    /**
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    public function renderView(): string
    {
        /** @var Clinique $clinique */
        $clinique = $this->loadObject();

        // Réductions de groupe
        $group_reduction = true;
        foreach (Clinique::getReductionsByCat() as $id_category => $reduction) {
            if (!GroupReduction::doesExist($this->object->id_group, $id_category)) {
                $group_reduction = false;
            }
        }

        $result = Db::getInstance(false)->executeS(
            'SELECT ch.*
                  FROM ' . _DB_PREFIX_ . CliniqueHolidays::TABLE . ' ch
                  WHERE ch.id_myvetshop_clinique = ' . (int) $clinique->id_myvetshop_clinique . '
                  ORDER BY ch.date_debut DESC
                  LIMIT 10');

        if (!is_array($result)) {
            $result = [];
        }

        $holidays = ObjectModel::hydrateCollection(
            'CliniqueHolidays', $result
        );

        /////////////////////////////////////////
        /// Livraison à domicile
        $carrier_home = new Carrier($this->object->id_carrier_home);
        $carrier_home_price = $carrier_home->getDeliveryPriceByWeight(50, 1);

        /////////////////////////////////////////
        /// URL logo / photo
        $photo_url = ImageManager::thumbnail(_PS_STORE_IMG_DIR_ . $clinique->id_store . '.jpg', 'store_' . (int) $clinique->id_store . '.' . $this->imageType, 150, $this->imageType, true, true);
        $logo_url = ImageManager::thumbnail(
            _PS_EMPLOYEE_IMG_DIR_ . $clinique->id_employee . '.jpg', 'employee_' . (int) $clinique->id_employee . '.' . $this->imageType, 150, $this->imageType, true, true
        );

        $this->tpl_view_vars = [
            'clinique' => $clinique,
            'carrier_home_price' => $carrier_home_price,
            'photo_url' => $photo_url,
            'logo_url' => $logo_url,
            'holidays' => $holidays,
            'carrier' => new Carrier($clinique->id_carrier),
            'carrier_home' => new Carrier($clinique->id_carrier_home),
            'warehouse' => new Warehouse($clinique->id_warehouse),
            'employee' => new Employee($clinique->id_employee),
            'group' => new Group($clinique->id_group),
            'group_reduction' => $group_reduction,
            'store' => new Store($clinique->id_store),
            'orders' => $clinique->getOrders(10, 0),
            'order_count' => $clinique->getOrderCount(),
            'customers' => $clinique->getCustomers(10, 0),
            'customer_count' => $clinique->getCustomerCount(),
        ];

        return parent::renderView();
    }

    /**
     * @return false|ObjectModel|void
     *
     * @throws PrestaShopException
     *
     * {@inheritDoc}
     */
    public function processSave()
    {
        /** @var Clinique $clinique */
        $clinique = parent::processSave();
        $this->object = $clinique;

        if (!$this->object instanceof Clinique) {
            return $this->object;
        }

        $db = Db::getInstance();

        /////////////////////////////////////////
        /// Jours de livraison
        $jours = ['lundi', 'mardi', 'mercredi', 'jeudi', 'vendredi', 'samedi', 'dimanche'];
        $jours_livraison = [];
        $this->object->code_privilege = strtoupper($this->object->code_privilege);

        foreach ($jours as $jour) {
            if (Tools::getValue('jours_livraison_' . $jour)) {
                $jours_livraison[] = $jour;
            }
        }
        $this->object->jours_livraison = implode(',', $jours_livraison);

        $this->object->save();

        /////////////////////////////////////////
        /// Mise à jour des différents objets

        /// Employe
        $employee = new Employee($this->object->id_employee);
        $employee->firstname = Tools::getValue('firstname');
        $employee->lastname = Tools::getValue('lastname');
        $employee->save();

        // Livraison à la clinique
        $db->execute(
            'UPDATE `' . _DB_PREFIX_ . 'carrier` SET `name` = "' . $db->escape($employee->firstname . ' ' . $employee->lastname) . '" WHERE `id_carrier` = '
            . (int) $this->object->id_carrier
        );
        if (trim($this->object->jours_livraison) && count($jours_livraison)) {
            $livraison_text = implode(', le ', array_slice($jours_livraison, 0, -1)) . ' et le ' . end($jours_livraison);
        } else {
            $livraison_text = 'Livraison en 2 à 4 jours ouvrés (hors corse)';
        }
        $db->execute(
            'UPDATE `' . _DB_PREFIX_ . 'carrier_lang` SET `delay` = "Livraison le ' . $db->escape($livraison_text) . '" WHERE `id_carrier` = '
            . (int) $this->object->id_carrier
        );

        // Livraison à domicile
        $carrier_home_price = Tools::getValue('carrier_home_price', 14);
        $db->execute(
            'UPDATE `' . _DB_PREFIX_ . 'delivery` SET `price` = ' . (float) $carrier_home_price . ' WHERE `id_carrier` = ' . (int) $this->object->id_carrier_home
        );

        // Groupe
        $group = new Group($this->object->id_group);
        $group->name = [Context::getContext()->language->id => $this->object->code_privilege];
        $group->save();

        // Association entre les carrier et les warehouse
        $db->execute(
            'DELETE FROM `' . _DB_PREFIX_ . 'warehouse_carrier`'
            . ' WHERE `id_carrier` IN (' . (int) $this->object->id_carrier . ', ' . (int) $this->object->id_carrier_home . ') AND `id_warehouse` != ' . (int) $this->object->id_warehouse
        );
        $db->executeS(
            'INSERT INTO `' . _DB_PREFIX_ . 'warehouse_carrier` (`id_carrier`, `id_warehouse`)'
            . ' VALUES (' . (int) $this->object->id_carrier . ', ' . (int) $this->object->id_warehouse . '), '
            . ' (' . (int) $this->object->id_carrier_home . ', ' . (int) $this->object->id_warehouse . ') '
            . ' ON DUPLICATE KEY UPDATE `id_warehouse` = ' . (int) $this->object->id_warehouse
        );

        // Réductions de groupe
        foreach (Clinique::getReductionsByCat() as $id_category => $reduction) {
            if (!GroupReduction::doesExist($this->object->id_group, $id_category)) {
                $groupeReduction = new GroupReduction();
                $groupeReduction->id_group = $this->object->id_group;
                $groupeReduction->id_category = $id_category;
                $groupeReduction->reduction = $reduction;
                $groupeReduction->save();
            }
        }

        // Boutique
        $latitude = Tools::getValue('latitude');
        $longitude = Tools::getValue('longitude');

        // S'il manque une coordonnée GPS, on va utiliser le GeoCoding Google Maps
        if (empty($latitude) || empty($longitude) || 0 == floatval($latitude) || 0 == floatval($longitude)) {
            $ch = curl_init();

            curl_setopt(
                $ch,
                CURLOPT_URL,
                'https://maps.google.com/maps/api/geocode/json?key=' . self::GMAP_API_KEY . '&address=' . urlencode(
                    Tools::getValue('address1') . ', ' . Tools::getValue('postcode') . ' ' . Tools::getValue('city')
                )
            );
            curl_setopt($ch, CURLOPT_HEADER, 0);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            $result = curl_exec($ch);

            if (is_string($result)) {
                $result = json_decode($result, true);
            }

            curl_close($ch);

            if (isset($result['results'][0]['geometry']['location'])) {
                $latitude = $result['results'][0]['geometry']['location']['lat'];
                $longitude = $result['results'][0]['geometry']['location']['lng'];
            }
        }

        $store = new Store($this->object->id_store);
        $store->name = $employee->firstname . ' ' . $employee->lastname;
        $store->address1 = Tools::getValue('address1');
        $store->address2 = Tools::getValue('address2');
        $store->postcode = Tools::getValue('postcode');
        $store->latitude = $latitude;
        $store->longitude = $longitude;
        $store->city = Tools::getValue('city');
        $store->hours = serialize(
            [
                Tools::getValue('hours_lundi'),
                Tools::getValue('hours_mardi'),
                Tools::getValue('hours_mercredi'),
                Tools::getValue('hours_jeudi'),
                Tools::getValue('hours_vendredi'),
                Tools::getValue('hours_samedi'),
                Tools::getValue('hours_dimanche'),
            ]
        );
        $store->website = Tools::getValue('website');  /* @phpstan-ignore-line */
        $store->save();

        // Autorisation d'accès à la catégorie accessoires
        $this->allowAccessoires($this->object->id_group);
        if ($this->object->id_group_rural) {
            $this->allowAccessoires($this->object->id_group_rural);
        }

        // Réduction rurale (cat_id 740)
        $rural_ids = [740];
        $last_selected_ids = [740];
        while (count($last_selected_ids)) {
            $ids = $last_selected_ids;
            $last_selected_ids = [];

            $result = $db->executeS('SELECT c.id_category FROM `' . _DB_PREFIX_ . 'category` c WHERE id_parent IN (' . implode(', ', $ids) . ')');
            if (!is_array($result)) {
                $result = [];
            }

            foreach ($result as $row) {
                $rural_ids[] = (int) $row['id_category'];
                $last_selected_ids[] = (int) $row['id_category'];
            }
        }
        $id_group_rural = (int) $this->object->id_group_rural;

        // Désactive l'accès aux catégories de la rurale pour les groupes qui ne sont pas des tarifs privilège
        $db->execute('DELETE FROM `' . _DB_PREFIX_ . 'category_group` WHERE `id_group` <= 3 AND `id_category` IN (' . implode(', ', $rural_ids) . ')');

        if ($this->object->rurale && !GroupReduction::doesExist($this->object->id_group_rural, 740)) {
            // Autorise le groupe a accéder à la catégorie
            $query = 'INSERT INTO `' . _DB_PREFIX_ . 'category_group` (`id_category`, `id_group`) VALUES ';
            $query .= implode(
                ', ',
                array_map(
                    function ($id_cat) use ($id_group_rural) {
                        return '(' . $id_cat . ', ' . $id_group_rural . ')';
                    }, $rural_ids
                )
            );
            $query .= ' ON DUPLICATE KEY UPDATE id_group = ' . $id_group_rural;
            $db->execute($query);

            // Mise en place de la réduction
            $groupeReduction = new GroupReduction();
            $groupeReduction->id_group = $this->object->id_group_rural;
            $groupeReduction->id_category = 740;
            $groupeReduction->reduction = 0.185;
            $groupeReduction->save();

            // Réductions de groupe
            foreach (Clinique::getReductionsByCat() as $id_category => $reduction) {
                if (!GroupReduction::doesExist($this->object->id_group_rural, $id_category)) {
                    $groupeReduction = new GroupReduction();
                    $groupeReduction->id_group = $this->object->id_group_rural;
                    $groupeReduction->id_category = $id_category;
                    $groupeReduction->reduction = $reduction;
                    $groupeReduction->save();
                }
            }

            Cache::clean('*');
            Tools::clearSmartyCache();
        } elseif (!$this->object->rurale && GroupReduction::doesExist($this->object->id_group_rural, 740)) {
            // Supprime la réduction
            $db->execute('DELETE FROM `' . _DB_PREFIX_ . 'category_group` WHERE `id_group` = ' . (int) $this->object->id_group_rural . ' AND `id_category` IN (' . implode(', ', $rural_ids) . ')');
            $db->execute('DELETE FROM `' . _DB_PREFIX_ . 'group_reduction` WHERE `id_group` = ' . (int) $this->object->id_group_rural . ' AND `id_category` = 740');

            Cache::clean('*');
            Tools::clearSmartyCache();
        }

        // Vérification de l'association livreur / groupes
        if ($this->object->id_group) {
            $count = $db->getValue('SELECT COUNT(*) FROM `' . _DB_PREFIX_ . 'carrier_group` WHERE `id_group` = ' . (int) $this->object->id_group);

            if ($count > 5) {
                $db->execute('DELETE FROM `' . _DB_PREFIX_ . 'carrier_group` WHERE `id_group` = ' . (int) $this->object->id_group);
            }
        }
        if ($this->object->id_group_rural) {
            $count = $db->getValue('SELECT COUNT(*) FROM `' . _DB_PREFIX_ . 'carrier_group` WHERE `id_group` = ' . (int) $this->object->id_group_rural);

            if ($count > 5) {
                $db->execute('DELETE FROM `' . _DB_PREFIX_ . 'carrier_group` WHERE `id_group` = ' . (int) $this->object->id_group_rural);
            }
        }

        // Association avec la livraison en clinique
        $carrier = new Carrier($this->object->id_carrier);
        $carrierGroups = array_map(
            function ($row) {
                return $row['id_group'];
            },
            $carrier->getGroups()
        );
        // Reset des groupes associés si trop de groupes sont présents
        if (count($carrierGroups) > 5) {
            $carrierGroups = [];
        }
        if (!in_array($this->object->id_group, $carrierGroups)) {
            $carrierGroups[] = $this->object->id_group;
            $carrier->setGroups($carrierGroups);
        }
        if ($this->object->id_group_rural && !in_array($this->object->id_group_rural, $carrierGroups)) {
            $carrierGroups[] = $this->object->id_group_rural;
            $carrier->setGroups($carrierGroups);
        }

        // Association avec la livraison à domicile
        $carrier = new Carrier($this->object->id_carrier_home);
        $carrierGroups = array_map(
            function ($row) {
                return $row['id_group'];
            },
            $carrier->getGroups()
        );
        // Reset des groupes associés si trop de groupes sont présents
        if (count($carrierGroups) > 5) {
            $carrierGroups = [];
        }
        if (!in_array($this->object->id_group, $carrierGroups)) {
            $carrierGroups[] = $this->object->id_group;
            $carrier->setGroups($carrierGroups);
        }
        if ($this->object->id_group_rural && !in_array($this->object->id_group_rural, $carrierGroups)) {
            $carrierGroups[] = $this->object->id_group_rural;
            $carrier->setGroups($carrierGroups);
        }
    }

    /**
     * Overload this method for custom checking
     *
     * @param mixed $id Object id used for deleting images
     *
     * @return bool
     *
     * {@inheritDoc}
     */
    protected function postImage($id): bool
    {
        $obj = $this->loadObject(true);
        if ($obj instanceof Clinique) {
            foreach ($this->fieldImageSettings as $image) {
                if (isset($image['name'], $image['dir'], $image['id_field'], $image['width'], $image['height'])) {
                    // Récupère l'ID depuis le champ qui va bien
                    $id = $obj->{$image['id_field']};
                    $this->uploadImage($id, $image['name'], $image['dir'] . '/', false, $image['width'], $image['height']);
                }
            }

            return true;
        }

        return false;
    }

    /**
     * Gestion des filtres sur les champs calculés
     */
    public function processFilter(): void
    {
        parent::processFilter();

        if ($this->_filterHaving) {
            $this->_filterHaving = str_replace(
                [
                    '`id_store`',
                    '`id_employee`',
                    '`id_warehouse`',
                ],
                [
                    "(SELECT CONCAT(s.address1, ' ', s.address1, ' ', s.postcode, ' ', s.city) FROM `" . _DB_PREFIX_ . 'store` s WHERE s.id_store = a.id_store )',
                    "(SELECT CONCAT(e.firstname, ' ', e.lastname) FROM `" . _DB_PREFIX_ . 'employee` e WHERE e.id_employee = a.id_employee )',
                    '(SELECT w.name FROM `' . _DB_PREFIX_ . 'warehouse` w WHERE w.id_warehouse = a.id_warehouse )',
                ],
                $this->_filterHaving
            );
        }
    }

    public function initContent(): void
    {
        if (!$this->viewAccess()) {
            $this->errors[] = Tools::displayError('You do not have permission to view this.');

            return;
        }
        if (isset($_GET['exportabandonmyvetshop_clinique'])) {
            /** @var Clinique $clinique */
            $clinique = $this->loadObject();

            $query = 'SELECT'
                . ' c.firstname AS prenom,'
                . ' c.lastname AS nom,'
                . ' c.email AS email,'
                . ' MAX(a.phone) AS telephone,'
                . ' MAX(a.phone_mobile) AS telephone_mobile,'
                . ' c.date_add AS date_inscription,'
                . ' COUNT(o.id_order) AS nb_commandes,'
                . ' MAX(o.invoice_date) AS date_derniere_commande'
                . ' FROM `' . _DB_PREFIX_ . 'customer` c'
                . ' LEFT JOIN `' . _DB_PREFIX_ . 'address` a ON a.id_customer = c.id_customer AND a.deleted = 0 AND a.active = 1'
                . ' LEFT JOIN `' . _DB_PREFIX_ . 'orders` o ON o.id_customer = c.id_customer'
                . ' WHERE c.id_default_group = ' . $clinique->id_group . ' AND c.active = 1'
                . ' GROUP BY c.id_customer'
                . ' HAVING MAX(o.invoice_date) < DATE_SUB(NOW(), INTERVAL 4 MONTH)'
                . ' ORDER BY MAX(o.invoice_date) DESC';

            header('Content-Type: application/json');
            header('Content-Disposition: attachment; filename="export_abandons_' . date('Y-m-d') . '.csv"');
            $stdout = fopen('php://temp ', 'w+');

            if (!$stdout) {
                throw new Exception('Error lors de l\'ouverture du fichier');
            }
            fwrite($stdout, "\xEF\xBB\xBF");
            $header = null;

            $result = Db::getInstance(false)->executeS($query);

            if (!is_array($result)) {
                $result = [];
            }
            foreach ($result as $row) {
                if (!$header) {
                    $header = array_keys($row);
                    fputcsv($stdout, $header, ';');
                }

                fputcsv($stdout, $row, ';');
            }

            rewind($stdout);
            fpassthru($stdout);
            fclose($stdout);
            exit();
        }

        parent::initContent();
    }
}
