<?php

declare(strict_types=1);

namespace NoahVet\Reef\Test\A_Unit\Security\User\Provider;

use NoahVet\Reef\Exception\NotJWTException;
use NoahVet\Reef\Factory\HttpClientFactoryInterface;
use NoahVet\Reef\Security\User\Provider\JWTReefOAuthUserProvider;
use NoahVet\Reef\Security\User\ReefOAuthUser;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Psr\Cache\CacheItemInterface;
use Psr\Cache\CacheItemPoolInterface;
use Psr\Http\Client\ClientInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\StreamInterface;
use SlopeIt\ClockMock\ClockMock;
use Symfony\Component\Cache\Adapter\ArrayAdapter;

class JWTReefOAuthUserProviderTest extends TestCase
{
    private JWTReefOAuthUserProvider $provider;

    /**
     * @var HttpClientFactoryInterface&MockObject
     */
    private HttpClientFactoryInterface $httpClientFactory;

    /**
     * @var CacheItemPoolInterface&MockObject
     */
    private CacheItemPoolInterface $cache;

    protected function setUp(): void
    {
        $this->httpClientFactory = $this->createMock(HttpClientFactoryInterface::class);
        $this->cache = $this->createMock(CacheItemPoolInterface::class);

        $this->provider = new JWTReefOAuthUserProvider($this->cache, 'base_url', $this->httpClientFactory);
    }

    protected function tearDown(): void
    {
        ClockMock::reset();
    }

    public function testLoadUserNoCache(): void
    {
        $provider = new JWTReefOAuthUserProvider(new ArrayAdapter(), 'base_url', $this->httpClientFactory);

        $bearerToken = 'eyJ0eXAiOiJKV1QiLCJhbGciOiJFUzI1NiIsImtpZCI6IjFlZTU1ZmU1LTZjMTUtNjZmMi04YmY3LWE5YmY'
            .'4MGVkZmVjOSJ9.eyJpc3MiOiJodHRwczovL3JlZWYtaWFtLnRyYWVmaWsubWV0IiwiYXVkIjoicmVlZjpic21BcGkiLCJp'
            .'YXQiOjE2OTUwNDQzMzIuNzcwMDE2LCJleHAiOjIwMTA0MDQzMzIsInN1YiI6IjkyYzI0ZjU3LWYzMzktNDZkNC1iYmE2LTJ'
            .'jZjY5NjkwYjFhMCIsInNjb3BlIjoiand0IHJlZWY6cmFiYml0bXEucmVhZDpyZWVmLyovcmVlZjpic21BcGkgcmVlZjpyYW'
            .'JiaXRtcS53cml0ZTpyZWVmLyovcmVlZjpic21BcGkgcmVlZjpyYWJiaXRtcS5jb25maWd1cmU6cmVlZi8qL3JlZWY6YnNtQ'
            .'XBpIiwiZW1haWwiOm51bGwsImVtYWlsX3ZlcmlmaWVkIjp0cnVlfQ.o74ayQtVt6yeQs_mJ4yRnxdffqgjXTCa3KeXmrzAP'
            .'xUl4e5VC4B60WE6-BXDK9v071ejappJakyEiow33nDdlQ';

        $httpClient = $this->createMock(ClientInterface::class);
        $jwtkeysResponse = $this->createMock(ResponseInterface::class);
        $jwtkeysResponseBody = $this->createMock(StreamInterface::class);
        $jwtkeysResponseBody
            ->method('getContents')
            ->willReturn('{"keys":[{"crv":"P-256","kty":"EC","alg":"ES256",'
                .'"x":"9KskQVLe6AjTBguliNKDQ61XMwbJ70bJ5kPpTK91kqs",'
                .'"y":"ECqtOuj6t9iOvGjzpcxNtzIrysijP88YJNFOH_nVLJg",'
                .'"use":"sig","kid":"1ee55fe5-6c15-66f2-8bf7-a9bf80edfec9"}]}')
        ;
        $jwtkeysResponse
            ->method('getBody')
            ->willReturn($jwtkeysResponseBody)
        ;
        $jwtkeysResponse
            ->method('getStatusCode')
            ->willReturn(200)
        ;

        $httpClient
            ->method('sendRequest')
            ->willReturn($jwtkeysResponse)
        ;

        $this->httpClientFactory->expects($this->once())
            ->method('create')
            ->willReturn($httpClient)
        ;

        $user = $provider->loadUser($bearerToken);

        $this->assertInstanceOf(ReefOAuthUser::class, $user);
        $this->assertEquals('reef:iam:principal:92c24f57-f339-46d4-bba6-2cf69690b1a0', $user->getUserIdentifier());
    }

    public function testLoadUserCached(): void
    {
        $provider = new JWTReefOAuthUserProvider($this->cache, 'base_url', $this->httpClientFactory);

        $bearerToken = 'eyJ0eXAiOiJKV1QiLCJhbGciOiJFUzI1NiIsImtpZCI6IjFlZTU1ZmU1LTZjMTUtNjZmMi04YmY3LWE5YmY'
            .'4MGVkZmVjOSJ9.eyJpc3MiOiJodHRwczovL3JlZWYtaWFtLnRyYWVmaWsubWV0IiwiYXVkIjoicmVlZjpic21BcGkiLCJp'
            .'YXQiOjE2OTUwNDQzMzIuNzcwMDE2LCJleHAiOjIwMTA0MDQzMzIsInN1YiI6IjkyYzI0ZjU3LWYzMzktNDZkNC1iYmE2LTJ'
            .'jZjY5NjkwYjFhMCIsInNjb3BlIjoiand0IHJlZWY6cmFiYml0bXEucmVhZDpyZWVmLyovcmVlZjpic21BcGkgcmVlZjpyYW'
            .'JiaXRtcS53cml0ZTpyZWVmLyovcmVlZjpic21BcGkgcmVlZjpyYWJiaXRtcS5jb25maWd1cmU6cmVlZi8qL3JlZWY6YnNtQ'
            .'XBpIiwiZW1haWwiOm51bGwsImVtYWlsX3ZlcmlmaWVkIjp0cnVlfQ.o74ayQtVt6yeQs_mJ4yRnxdffqgjXTCa3KeXmrzAP'
            .'xUl4e5VC4B60WE6-BXDK9v071ejappJakyEiow33nDdlQ';

        $cacheItem = $this->createMock(CacheItemInterface::class);

        $cacheItem
            ->method('isHit')
            ->willReturn(true)
        ;

        $cacheItem
            ->method('get')
            ->willReturn([
                'keys' => [
                    [
                        'crv' => 'P-256',
                        'kty' => 'EC',
                        'alg' => 'ES256',
                        'x' => '9KskQVLe6AjTBguliNKDQ61XMwbJ70bJ5kPpTK91kqs',
                        'y' => 'ECqtOuj6t9iOvGjzpcxNtzIrysijP88YJNFOH_nVLJg',
                        'use' => 'sig',
                        'kid' => '1ee55fe5-6c15-66f2-8bf7-a9bf80edfec9',
                    ],
                ],
            ])
        ;

        $this->cache
            ->method('getItem')
            ->with('iam-keys')
            ->willReturn($cacheItem)
        ;

        $user = $provider->loadUser($bearerToken);

        $this->assertInstanceOf(ReefOAuthUser::class, $user);
        $this->assertEquals('reef:iam:principal:92c24f57-f339-46d4-bba6-2cf69690b1a0', $user->getUserIdentifier());
    }

    public function testLoadUserEmptyToken(): void
    {
        $provider = new JWTReefOAuthUserProvider($this->cache, 'base_url', $this->httpClientFactory);

        $this->expectException(NotJWTException::class);

        $provider->loadUser('');
    }

    public function testLoadUserIncorrectToken(): void
    {
        $provider = new JWTReefOAuthUserProvider($this->cache, 'base_url', $this->httpClientFactory);

        $this->expectException(NotJWTException::class);

        $provider->loadUser('incorrect_token');
    }

    public function testLoadUserExpired(): void
    {
        $provider = new JWTReefOAuthUserProvider(new ArrayAdapter(), 'base_url', $this->httpClientFactory);

        ClockMock::freeze(new \DateTimeImmutable('2050-04-06 00:00:00'));

        $bearerToken = 'eyJ0eXAiOiJKV1QiLCJhbGciOiJFUzI1NiIsImtpZCI6IjFlZTU1ZmU1LTZjMTUtNjZmMi04YmY3LWE5YmY'
            .'4MGVkZmVjOSJ9.eyJpc3MiOiJodHRwczovL3JlZWYtaWFtLnRyYWVmaWsubWV0IiwiYXVkIjoicmVlZjpic21BcGkiLCJp'
            .'YXQiOjE2OTUwNDQzMzIuNzcwMDE2LCJleHAiOjIwMTA0MDQzMzIsInN1YiI6IjkyYzI0ZjU3LWYzMzktNDZkNC1iYmE2LTJ'
            .'jZjY5NjkwYjFhMCIsInNjb3BlIjoiand0IHJlZWY6cmFiYml0bXEucmVhZDpyZWVmLyovcmVlZjpic21BcGkgcmVlZjpyYW'
            .'JiaXRtcS53cml0ZTpyZWVmLyovcmVlZjpic21BcGkgcmVlZjpyYWJiaXRtcS5jb25maWd1cmU6cmVlZi8qL3JlZWY6YnNtQ'
            .'XBpIiwiZW1haWwiOm51bGwsImVtYWlsX3ZlcmlmaWVkIjp0cnVlfQ.o74ayQtVt6yeQs_mJ4yRnxdffqgjXTCa3KeXmrzAP'
            .'xUl4e5VC4B60WE6-BXDK9v071ejappJakyEiow33nDdlQ';

        $httpClient = $this->createMock(ClientInterface::class);
        $jwtkeysResponse = $this->createMock(ResponseInterface::class);
        $jwtkeysResponseBody = $this->createMock(StreamInterface::class);
        $jwtkeysResponseBody
            ->method('getContents')
            ->willReturn('{"keys":[{"crv":"P-256","kty":"EC","alg":"ES256",'
                .'"x":"9KskQVLe6AjTBguliNKDQ61XMwbJ70bJ5kPpTK91kqs",'
                .'"y":"ECqtOuj6t9iOvGjzpcxNtzIrysijP88YJNFOH_nVLJg",'
                .'"use":"sig","kid":"1ee55fe5-6c15-66f2-8bf7-a9bf80edfec9"}]}')
        ;
        $jwtkeysResponse
            ->method('getBody')
            ->willReturn($jwtkeysResponseBody)
        ;
        $jwtkeysResponse
            ->method('getStatusCode')
            ->willReturn(200)
        ;

        $httpClient
            ->method('sendRequest')
            ->willReturn($jwtkeysResponse)
        ;

        $this->httpClientFactory->expects($this->once())
            ->method('create')
            ->willReturn($httpClient)
        ;

        $user = $provider->loadUser($bearerToken);

        $this->assertNUll($user);
    }
}
