<?php

declare(strict_types=1);

namespace NoahVet\Reef\Security\User\Provider;

use NoahVet\Reef\Factory\ClientFactoryInterface;
use NoahVet\Reef\Jane\Exception\TokeninfoUnauthorizedException;
use NoahVet\Reef\Jane\Exception\UserinfoUnauthorizedException;
use NoahVet\Reef\Jane\Model\OauthV2TokenInfoTokenGetResponse200;
use NoahVet\Reef\Jane\Model\UserinfoGetResponse200;
use NoahVet\Reef\Security\User\ReefOAuthUser;
use Psr\Cache\CacheItemPoolInterface;
use Symfony\Component\Cache\Adapter\ArrayAdapter;

class UserInfoReefOAuthUserProvider implements ReefOAuthUserProviderInterface
{
    /**
     * Default : cache user for 30 seconds.
     */
    public const REEF_USER_CACHE_DURATION = 30;

    protected CacheItemPoolInterface $cache;

    public function __construct(
        private readonly ClientFactoryInterface $clientFactory,
    ) {
        $this->cache = new ArrayAdapter();
    }

    public function loadUser(string $bearerToken): ?ReefOAuthUser
    {
        $userInfo = $this->getUserInfo($bearerToken);

        return $userInfo ? ReefOAuthUser::createFromUserinfo($userInfo) : null;
    }

    public function getCache(): CacheItemPoolInterface
    {
        return $this->cache;
    }

    public function getTokenExpiresAt(string $bearerToken): ?\DateTimeImmutable
    {
        $cacheItem = $this->cache->getItem('tokeninfo-'.$bearerToken);

        if (!$cacheItem->isHit()) {
            $client = $this->clientFactory->create($bearerToken);

            try {
                /** @var OauthV2TokenInfoTokenGetResponse200|null $tokeninfo */
                $tokeninfo = $client->tokeninfo(['token' => $bearerToken]);

                $cacheItem->set($tokeninfo);
                $cacheItem->expiresAfter(self::REEF_USER_CACHE_DURATION);
                $this->cache->save($cacheItem);
            } catch (TokeninfoUnauthorizedException) {
                // Invalid token
                return null;
            }
        } else {
            $tokeninfo = $cacheItem->get();
        }

        if (null === $tokeninfo || !$tokeninfo instanceof OauthV2TokenInfoTokenGetResponse200) {
            return null;
        }

        return \DateTimeImmutable::createFromMutable($tokeninfo->getExpiresAt());
    }

    public function setCache(?CacheItemPoolInterface $cache): void
    {
        $this->cache = $cache ?? $this->cache;
    }

    protected function getUserInfo(string $bearerToken): ?UserinfoGetResponse200
    {
        $cacheItem = $this->cache->getItem('userinfo-'.$bearerToken);

        if (!$cacheItem->isHit()) {
            $client = $this->clientFactory->create($bearerToken);

            try {
                /** @var UserinfoGetResponse200|null $userInfo */
                $userInfo = $client->userinfo();

                $cacheItem->set($userInfo);
                $cacheItem->expiresAfter(self::REEF_USER_CACHE_DURATION);
                $this->cache->save($cacheItem);
            } catch (UserinfoUnauthorizedException) {
                // Invalid token
                return null;
            }
        } else {
            $userInfo = $cacheItem->get();
        }

        if (null === $userInfo || $userInfo instanceof UserinfoGetResponse200) {
            return $userInfo;
        }

        return null;
    }
}
