<?php

declare(strict_types=1);

namespace NoahVet\Reef\Security;

use NoahVet\Reef\Exception\NotJWTException;
use NoahVet\Reef\Security\Authentication\Passport\Badge\TokenExpirationBadge;
use NoahVet\Reef\Security\Authentication\Token\ReefOAuthToken;
use NoahVet\Reef\Security\User\Provider\ReefOAuthUserProviderInterface;
use NoahVet\Reef\Security\User\ReefOAuthUser;
use Psr\Cache\CacheItemPoolInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\Security\Http\Authenticator\AbstractAuthenticator;
use Symfony\Component\Security\Http\Authenticator\Passport\Badge\UserBadge;
use Symfony\Component\Security\Http\Authenticator\Passport\Passport;
use Symfony\Component\Security\Http\Authenticator\Passport\SelfValidatingPassport;

class ReefOAuthAuthenticator extends AbstractAuthenticator implements ReefOAuthAuthenticatorInterface
{
    /**
     * @param iterable<ReefOAuthUserProviderInterface> $userProviders
     */
    public function __construct(
        private readonly iterable $userProviders,
    ) {
    }

    public function supports(Request $request): ?bool
    {
        return $request->headers->has('Authorization')
            && \str_starts_with($request->headers->get('Authorization') ?? '', 'Bearer ');
    }

    public function authenticate(Request $request): Passport
    {
        $oauthToken = \trim(\substr($request->headers->get('Authorization') ?? '', 7));

        return new SelfValidatingPassport(
            new UserBadge($oauthToken, [$this, 'loadUser']),
            [new TokenExpirationBadge($oauthToken, [$this, 'getTokenExpiresAt'])],
        );
    }

    public function createToken(Passport $passport, string $firewallName): TokenInterface
    {
        /** @var UserBadge $userBadge */
        $userBadge = $passport->getBadge(UserBadge::class);
        /** @var ReefOAuthUser $user */
        $user = $userBadge->getUser();

        /** @var TokenExpirationBadge $tokenExpirationBadge */
        $tokenExpirationBadge = $passport->getBadge(TokenExpirationBadge::class);

        return new ReefOAuthToken($firewallName, $userBadge->getUserIdentifier(), $user, $tokenExpirationBadge->getExpirationDate());
    }

    public function onAuthenticationSuccess(Request $request, TokenInterface $token, string $firewallName): ?Response
    {
        return null;
    }

    public function onAuthenticationFailure(Request $request, AuthenticationException $exception): ?Response
    {
        if (\str_starts_with($request->headers->get('Authorization') ?? '', 'Bearer ')) {
            return new JsonResponse(['error' => 'Invalid bearer token provided'], Response::HTTP_FORBIDDEN);
        } else {
            return new JsonResponse(['error' => 'No bearer token provided'], Response::HTTP_UNAUTHORIZED);
        }
    }

    public function getTokenExpiresAt(string $bearerToken): ?\DateTimeImmutable
    {
        foreach ($this->userProviders as $userProvider) {
            try {
                return $userProvider->getTokenExpiresAt($bearerToken);
            } catch (NotJWTException) {
                /* Error inhibitor : Token can safely be not of type JWT */
            }
        }

        return null;
    }

    public function loadUser(string $bearerToken): ?ReefOAuthUser
    {
        foreach ($this->userProviders as $userProvider) {
            try {
                return $userProvider->loadUser($bearerToken);
            } catch (NotJWTException) {
                /* Error inhibitor : Token can safely be not of type JWT */
            }
        }

        return null;
    }

    /**
     * @deprecated
     */
    public function setCache(?CacheItemPoolInterface $cache): void
    {
    }

    /**
     * @deprecated
     */
    public function getCache(): ?CacheItemPoolInterface
    {
        return null;
    }
}
