<?php

declare(strict_types=1);

namespace NoahVet\Reef\Jane\Endpoint;

class PutDefinitionClientItem extends \NoahVet\Reef\Jane\Runtime\Client\BaseEndpoint implements \NoahVet\Reef\Jane\Runtime\Client\Endpoint
{
    use \NoahVet\Reef\Jane\Runtime\Client\EndpointTrait;

    protected $slug;

    /**
     * Update client resources types and allowed permissions.
     *
     * @param string $slug             Resource identifier
     * @param array  $headerParameters {
     *
     * @var string $X-Reef-Client-ID Public OAuth Client ID
     * @var string $X-Reef-Client-HMAC HMAC computed using hash_hmac('sha256', bearer, clientSecret)
     *             }
     */
    public function __construct(string $slug, \NoahVet\Reef\Jane\Model\ServiceDefinition $requestBody, array $headerParameters = [])
    {
        $this->slug = $slug;
        $this->body = $requestBody;
        $this->headerParameters = $headerParameters;
    }

    public function getMethod(): string
    {
        return 'PUT';
    }

    public function getUri(): string
    {
        return \str_replace(['{slug}'], [$this->slug], '/api/clients/{slug}/definition');
    }

    public function getBody(\Symfony\Component\Serializer\SerializerInterface $serializer, $streamFactory = null): array
    {
        if ($this->body instanceof \NoahVet\Reef\Jane\Model\ServiceDefinition) {
            return [['Content-Type' => ['application/json']], $serializer->serialize($this->body, 'json')];
        }

        return [[], null];
    }

    public function getExtraHeaders(): array
    {
        return ['Accept' => ['application/json']];
    }

    public function getAuthenticationScopes(): array
    {
        return ['BearerAuth'];
    }

    protected function getHeadersOptionsResolver(): \Symfony\Component\OptionsResolver\OptionsResolver
    {
        $optionsResolver = parent::getHeadersOptionsResolver();
        $optionsResolver->setDefined(['X-Reef-Client-ID', 'X-Reef-Client-HMAC']);
        $optionsResolver->setRequired([]);
        $optionsResolver->setDefaults([]);
        $optionsResolver->addAllowedTypes('X-Reef-Client-ID', ['string']);
        $optionsResolver->addAllowedTypes('X-Reef-Client-HMAC', ['string']);

        return $optionsResolver;
    }

    /**
     * @return null
     *
     * @throws \NoahVet\Reef\Jane\Exception\PutDefinitionClientItemBadRequestException
     * @throws \NoahVet\Reef\Jane\Exception\PutDefinitionClientItemUnprocessableEntityException
     * @throws \NoahVet\Reef\Jane\Exception\PutDefinitionClientItemNotFoundException
     * @throws \NoahVet\Reef\Jane\Exception\UnexpectedStatusCodeException
     */
    protected function transformResponseBody(\Psr\Http\Message\ResponseInterface $response, \Symfony\Component\Serializer\SerializerInterface $serializer, ?string $contentType = null)
    {
        $status = $response->getStatusCode();
        $body = (string) $response->getBody();
        if ((null === $contentType) === false && (200 === $status && false !== \mb_strpos($contentType, 'application/json'))) {
            return \json_decode($body);
        }
        if (400 === $status) {
            throw new \NoahVet\Reef\Jane\Exception\PutDefinitionClientItemBadRequestException($response);
        }
        if (422 === $status) {
            throw new \NoahVet\Reef\Jane\Exception\PutDefinitionClientItemUnprocessableEntityException($response);
        }
        if (404 === $status) {
            throw new \NoahVet\Reef\Jane\Exception\PutDefinitionClientItemNotFoundException($response);
        }
        if ((null === $contentType) === false && (304 === $status && false !== \mb_strpos($contentType, 'application/json'))) {
            return \json_decode($body);
        }
        throw new \NoahVet\Reef\Jane\Exception\UnexpectedStatusCodeException($status, $body);
    }
}
