<?php

declare(strict_types=1);

namespace NoahVet\Reef\Test\A_Unit\Security\User\Provider;

use NoahVet\Reef\Factory\ClientFactoryInterface;
use NoahVet\Reef\Jane\Client;
use NoahVet\Reef\Jane\Exception\UserinfoUnauthorizedException;
use NoahVet\Reef\Jane\Model\UserinfoGetResponse200;
use NoahVet\Reef\Security\User\Provider\UserInfoReefOAuthUserProvider;
use NoahVet\Reef\Security\User\ReefOAuthUser;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Psr\Cache\CacheItemInterface;
use Psr\Cache\CacheItemPoolInterface;
use Symfony\Component\Cache\Adapter\ArrayAdapter;

class UserInfoReefOAuthUserProviderTest extends TestCase
{
    /**
     * @var ClientFactoryInterface&MockObject
     */
    private ClientFactoryInterface $clientFactory;

    /**
     * @var CacheItemPoolInterface&MockObject
     */
    private CacheItemPoolInterface $cache;

    private UserInfoReefOAuthUserProvider $userInfoReefOAuthUserProvider;

    protected function setUp(): void
    {
        $this->clientFactory = $this->createMock(ClientFactoryInterface::class);
        $this->cache = $this->createMock(CacheItemPoolInterface::class);

        $this->userInfoReefOAuthUserProvider = new UserInfoReefOAuthUserProvider($this->clientFactory);
        $this->userInfoReefOAuthUserProvider->setCache($this->cache);
    }

    public function testLoadUserNoCache(): void
    {
        $this->userInfoReefOAuthUserProvider->setCache(new ArrayAdapter());
        $client = $this->createMock(Client::class);

        $userInfo = $this->createMock(UserinfoGetResponse200::class);

        $userInfo
            ->method('getIdentifier')
            ->willReturn('92c24f57-f339-46d4-bba6-2cf69690b1a0')
        ;

        $userInfo
            ->method('getEmail')
            ->willReturn('mail@mail.com')
        ;

        $client
            ->method('userinfo')
            ->willReturn($userInfo)
        ;

        $this->clientFactory
            ->method('create')
            ->with('bearer')
            ->willReturn($client)
        ;

        $result = $this->userInfoReefOAuthUserProvider->loadUser('bearer');

        $this->assertInstanceOf(ReefOAuthUser::class, $result);
        $this->assertEquals('reef:iam:principal:92c24f57-f339-46d4-bba6-2cf69690b1a0', $result->getUserIdentifier());
    }

    public function testUserCached(): void
    {
        $cachedUserInfo = $this->createMock(UserinfoGetResponse200::class);
        $cachedUserInfo->method('getIdentifier')->willReturn('92c24f57-f339-46d4-bba6-2cf69690b1a0');
        $cachedUserInfo->method('getEmail')->willReturn('mail@mail.com');

        $cacheItem = $this->createMock(CacheItemInterface::class);

        $cacheItem
            ->method('isHit')
            ->willReturn(true)
        ;

        $cacheItem
            ->method('get')
            ->willReturn($cachedUserInfo)
        ;

        $this->cache
            ->method('getItem')
            ->with('userinfo-bearer')
            ->willReturn($cacheItem)
        ;

        $result = $this->userInfoReefOAuthUserProvider->loadUser('bearer');

        $this->assertInstanceOf(ReefOAuthUser::class, $result);
        $this->assertEquals('reef:iam:principal:92c24f57-f339-46d4-bba6-2cf69690b1a0', $result->getUserIdentifier());
    }

    public function testUserNotFound(): void
    {
        $this->userInfoReefOAuthUserProvider->setCache(new ArrayAdapter());
        $client = $this->createMock(Client::class);

        $client
            ->method('userinfo')
            ->willThrowException($this->createMock(UserinfoUnauthorizedException::class))
        ;

        $this->clientFactory
            ->method('create')
            ->with('bearer')
            ->willReturn($client)
        ;

        $result = $this->userInfoReefOAuthUserProvider->loadUser('bearer');

        $this->assertNull($result);
    }
}
