<?php

declare(strict_types=1);

namespace NoahVet\Reef\Security\User\Provider;

use Firebase\JWT\ExpiredException;
use Firebase\JWT\JWK;
use Firebase\JWT\JWT;
use Firebase\JWT\Key;
use NoahVet\Reef\Exception\NotJWTException;
use NoahVet\Reef\Exception\OAuthRequestException;
use NoahVet\Reef\Factory\HttpClientFactoryInterface;
use NoahVet\Reef\Security\User\ReefOAuthUser;
use Psr\Cache\CacheItemPoolInterface;

class JWTReefOAuthUserProvider implements ReefOAuthUserProviderInterface
{
    /**
     * @var array<string, Key>|null
     */
    private ?array $iamKeys = null;

    /**
     * @var array<string, \stdClass|null>
     */
    private array $decodedTokens = [];

    public function __construct(
        private readonly CacheItemPoolInterface $cache,
        private readonly string $reefOAuthBaseUrl,
        private readonly HttpClientFactoryInterface $httpClientFactory,
    ) {
    }

    /**
     * @throws NotJWTException
     * @throws OAuthRequestException
     */
    public function loadUser(string $bearerToken): ?ReefOAuthUser
    {
        $token = $this->decodeToken($bearerToken);

        if (null === $token) {
            return null;
        }

        return new ReefOAuthUser(
            (string) $token->sub,
            $token->email ?? null,
        );
    }

    public function getTokenExpiresAt(string $bearerToken): ?\DateTimeImmutable
    {
        $token = $this->decodeToken($bearerToken);

        if (null === $token) {
            return null;
        }

        $exp = (int) $token->exp;

        return new \DateTimeImmutable('@'.$exp);
    }

    /**
     * @return array<string, Key>
     */
    public function getIAMPublicKeys(): array
    {
        if (null === $this->iamKeys) {
            $cacheItem = $this->cache->getItem('iam-keys');

            if ($cacheItem->isHit()) {
                $keyDetails = $cacheItem->get();
            } else {
                $client = $this->httpClientFactory->create();

                $openIdKeysResponse = $client->sendRequest(
                    $this->httpClientFactory->getRequestFactory()->createRequest(
                        'GET',
                        $this->reefOAuthBaseUrl.'/openid/jwt-keys.json',
                    ),
                );

                if (200 !== $openIdKeysResponse->getStatusCode()) {
                    throw new OAuthRequestException("Can't load IAM's public keys");
                }

                $keyDetails = \json_decode($openIdKeysResponse->getBody()->getContents(), true);
            }

            if (!\is_array($keyDetails) || !\is_array($keyDetails['keys'] ?? null)) {
                if ($cacheItem->isHit()) {
                    // Force cache expiration
                    $cacheItem->expiresAfter(-1);
                    $this->cache->save($cacheItem);
                }
                throw new OAuthRequestException('IAM payload is invalid');
            }

            $this->iamKeys = JWK::parseKeySet($keyDetails);

            // Parse is OK, we can safely cache key details
            if (isset($openIdKeysResponse)) {
                $cacheHeader = $openIdKeysResponse->getHeader('Cache-Control')[0] ?? null;

                if (\is_string($cacheHeader) && \preg_match('#max-age=(\d+)#', $cacheHeader, $matches)) {
                    $cacheItem->set($keyDetails);
                    $cacheItem->expiresAfter((int) $matches[1]);
                    $this->cache->save($cacheItem);
                }
            }
        }

        return $this->iamKeys;
    }

    protected function decodeToken(string $bearerToken): ?\stdClass
    {
        if (!\array_key_exists($bearerToken, $this->decodedTokens)) {
            if (empty($bearerToken)) {
                throw new NotJWTException();
            }

            $tks = \explode('.', $bearerToken);
            if (3 !== \count($tks)) {
                throw new NotJWTException();
            }

            try {
                $decodedToken = JWT::decode($bearerToken, $this->getIAMPublicKeys());
            } catch (ExpiredException) {
                return null;
            } catch (\UnexpectedValueException) {
                throw new NotJWTException();
            }

            $this->decodedTokens[$bearerToken] = $decodedToken;
        }

        return $this->decodedTokens[$bearerToken];
    }
}
