<?php

declare(strict_types=1);

namespace NoahVet\Reef\Jane\Endpoint;

class GetAllManufacturerUser extends \NoahVet\Reef\Jane\Runtime\Client\BaseEndpoint implements \NoahVet\Reef\Jane\Runtime\Client\Endpoint
{
    use \NoahVet\Reef\Jane\Runtime\Client\EndpointTrait;

    /**
     * List Manufacturer Users.
     * A Manufacturer User s a member of the Manufacturer.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $department[] Filter by veterinary establishment department
     * @var array $jobTitle[] Filter by jobTitle
     * @var array $userId[] Filter by userId
     * @var array $manufacturerId[] Filter by manufacturerId
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     */
    public function __construct(array $queryParameters = [], array $headerParameters = [])
    {
        $this->queryParameters = $queryParameters;
        $this->headerParameters = $headerParameters;
    }

    public function getMethod(): string
    {
        return 'GET';
    }

    public function getUri(): string
    {
        return '/v1/manufacturer_users';
    }

    public function getBody(\Symfony\Component\Serializer\SerializerInterface $serializer, $streamFactory = null): array
    {
        return [[], null];
    }

    public function getExtraHeaders(): array
    {
        return ['Accept' => ['application/json']];
    }

    public function getAuthenticationScopes(): array
    {
        return ['BearerAuth'];
    }

    protected function getQueryOptionsResolver(): \Symfony\Component\OptionsResolver\OptionsResolver
    {
        $optionsResolver = parent::getQueryOptionsResolver();
        $optionsResolver->setDefined(['page', 'limit', 'id', 'department', 'jobTitle', 'userId', 'manufacturerId', 'fromCreatedDate', 'fromUpdatedDate', 'toCreatedDate', 'toUpdatedDate', 'sortCreatedDate', 'sortUpdatedDate']);
        $optionsResolver->setRequired([]);
        $optionsResolver->setDefaults([]);
        $optionsResolver->addAllowedTypes('page', ['int']);
        $optionsResolver->addAllowedTypes('limit', ['int']);
        $optionsResolver->addAllowedTypes('id', ['array']);
        $optionsResolver->addAllowedTypes('department', ['array']);
        $optionsResolver->addAllowedTypes('jobTitle', ['array']);
        $optionsResolver->addAllowedTypes('userId', ['array']);
        $optionsResolver->addAllowedTypes('manufacturerId', ['array']);
        $optionsResolver->addAllowedTypes('fromCreatedDate', ['array']);
        $optionsResolver->addAllowedTypes('fromUpdatedDate', ['array', 'null']);
        $optionsResolver->addAllowedTypes('toCreatedDate', ['array', 'null']);
        $optionsResolver->addAllowedTypes('toUpdatedDate', ['array', 'null']);
        $optionsResolver->addAllowedTypes('sortCreatedDate', ['string']);
        $optionsResolver->addAllowedTypes('sortUpdatedDate', ['string']);

        return $optionsResolver;
    }

    protected function getHeadersOptionsResolver(): \Symfony\Component\OptionsResolver\OptionsResolver
    {
        $optionsResolver = parent::getHeadersOptionsResolver();
        $optionsResolver->setDefined(['Accept-Language']);
        $optionsResolver->setRequired([]);
        $optionsResolver->setDefaults([]);
        $optionsResolver->addAllowedTypes('Accept-Language', ['string']);

        return $optionsResolver;
    }

    /**
     * @return \NoahVet\Reef\Jane\Model\ManufacturerUserResponseItem[]
     *
     * @throws \NoahVet\Reef\Jane\Exception\GetAllManufacturerUserBadRequestException
     * @throws \NoahVet\Reef\Jane\Exception\UnexpectedStatusCodeException
     */
    protected function transformResponseBody(\Psr\Http\Message\ResponseInterface $response, \Symfony\Component\Serializer\SerializerInterface $serializer, ?string $contentType = null)
    {
        $status = $response->getStatusCode();
        $body = (string) $response->getBody();
        if ((null === $contentType) === false && (200 === $status && false !== \mb_strpos($contentType, 'application/json'))) {
            return $serializer->deserialize($body, 'NoahVet\\Reef\\Jane\\Model\\ManufacturerUserResponseItem[]', 'json');
        }
        if ((null === $contentType) === false && (206 === $status && false !== \mb_strpos($contentType, 'application/json'))) {
            return $serializer->deserialize($body, 'NoahVet\\Reef\\Jane\\Model\\ManufacturerUserResponseItem[]', 'json');
        }
        if (400 === $status) {
            throw new \NoahVet\Reef\Jane\Exception\GetAllManufacturerUserBadRequestException($response);
        }
        throw new \NoahVet\Reef\Jane\Exception\UnexpectedStatusCodeException($status, $body);
    }
}
