<?php

declare(strict_types=1);

namespace NoahVet\Reef\Cache\Factory;

use NoahVet\Reef\Cache\Exception\CacheAdapterNotSupportedException;
use Psr\Cache\CacheItemPoolInterface;
use Symfony\Component\Cache\Adapter\ArrayAdapter;
use Symfony\Component\Cache\Adapter\NullAdapter;
use Symfony\Component\Cache\Adapter\RedisAdapter;

class ReefCacheFactory implements ReefCacheFactoryInterface
{
    /**
     * @var array<CacheItemPoolInterface>
     */
    private readonly array $cachePools;

    /**
     * @param iterable<CacheItemPoolInterface> $cachePools
     */
    public function __construct(
        iterable $cachePools,
    ) {
        $this->cachePools = $cachePools instanceof \Traversable ? \iterator_to_array($cachePools) : $cachePools;
    }

    /**
     * @throws CacheAdapterNotSupportedException
     */
    public function createCacheAdapter(string $cacheUrl): CacheItemPoolInterface
    {
        $scheme = \parse_url($cacheUrl, \PHP_URL_SCHEME);
        if (false === $scheme || null === $scheme) {
            $scheme = $cacheUrl;
        }

        return match ($scheme) {
            'cache.app' => $this->getAdapterByServiceName('service://cache.app'), // Retro-compatibility
            'array' => new ArrayAdapter(),
            'null' => new NullAdapter(),
            'redis' => $this->createRedisAdapter($cacheUrl),
            'service' => $this->getAdapterByServiceName($cacheUrl),
            default => throw new CacheAdapterNotSupportedException($cacheUrl),
        };
    }

    /**
     * @psalm-suppress UndefinedClass
     */
    private function createRedisAdapter(string $cacheUrl): CacheItemPoolInterface
    {
        return new RedisAdapter(RedisAdapter::createConnection($cacheUrl, ['lazy' => true]));
    }

    private function getAdapterByServiceName(string $cacheUrl): CacheItemPoolInterface
    {
        $serviceName = \parse_url($cacheUrl, \PHP_URL_HOST);

        if (!\is_string($serviceName) || !\array_key_exists($serviceName, $this->cachePools)) {
            throw new CacheAdapterNotSupportedException($cacheUrl);
        }

        return $this->cachePools[$serviceName];
    }
}
