<?php

declare(strict_types=1);

namespace NoahVet\Reef\Controller\Common;

use NoahVet\Reef\Controller\AbstractController;
use NoahVet\Reef\Factory\RequestContext\RequestContextFactoryInterface;
use NoahVet\Reef\Mapper\Common\One\Create\AbstractCreateMapperInterface;
use NoahVet\Reef\Mapper\Common\Search\AbstractSearchMapperInterface;
use NoahVet\Reef\Paginator\HttpPaginatorInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Response;

/**
 * @template EntityType of object
 * @template RequestDtoType of object
 * @template ResponseDtoType of object
 */
abstract class AbstractSearchController extends AbstractController
{
    public function __construct(
        protected readonly HttpPaginatorInterface $httpPaginator,
        protected readonly RequestContextFactoryInterface $requestContextFactory,
    ) {
    }

    /**
     * @param AbstractCreateMapperInterface<EntityType, RequestDtoType, ResponseDtoType> $mapper
     * @param class-string<RequestDtoType>                                               $requestDtoFqcn
     *
     * @return EntityType
     */
    protected function deserializeRequest(
        AbstractCreateMapperInterface $mapper,
        string $requestDtoFqcn,
    ): object {
        return $mapper->mapRequestDtoToEntity(
            $mapper->mapDataToRequestDto(
                $this->request->getContent(),
                $requestDtoFqcn,
            ),
        );
    }

    /**
     * @param array<int, mixed>|\ArrayObject<(int|string), mixed>|bool|float|int|string|null $data
     */
    protected function getResponse(array|\ArrayObject|bool|float|int|string|null $data): Response
    {
        if (0 < $this->httpPaginator->getInformation()['last']) {
            $this->headers = \array_merge($this->headers, $this->httpPaginator->getContentRangeHeader('resources'));
        }

        return new JsonResponse(
            $data,
            0 < $this->httpPaginator->getInformation()['last']
                ? Response::HTTP_PARTIAL_CONTENT
                : Response::HTTP_OK,
            $this->headers,
        );
    }

    /**
     * @param AbstractSearchMapperInterface<EntityType, RequestDtoType, ResponseDtoType> $mapper
     * @param class-string<EntityType>                                                   $entityFqcn
     * @param class-string<ResponseDtoType>                                              $responseDtoFqcn
     *
     * @return array<int, mixed>|\ArrayObject<(int|string), mixed>|bool|float|int|string|null
     */
    protected function serializeResponse(
        AbstractSearchMapperInterface $mapper,
        string $entityFqcn,
        string $responseDtoFqcn,
    ): array|\ArrayObject|bool|float|int|string|null {
        return $mapper->normalizeResponseDto(
            $mapper->mapDataToResponseDto(
                $this->httpPaginator->getResult(
                    $this->request,
                    $entityFqcn,
                    $this->requestContextFactory->createContextFromRequest(
                        $this->request,
                        $this->whitelist,
                    ),
                ),
                $responseDtoFqcn,
            ),
        );
    }
}
