# reef-api-client

Ce client permet de connecter simplement un projet Symfony au reef.

## Installation
Add the satis repository to composer.json :
```json
{
  "config": {
    "repositories": [
      {
        "type": "composer",
        "url": "https://satis.vetinweb.com"
      }
    ]
  }
}
```

Then require the package

```shell
composer require noahvet/reef-api-client

# Add project-specific client (See client list bellow)
composer require noahvet/reef-bsm-client
```

## Usage

### direct (no integration)
Create a new client :

```php
$clientFactory = new \NoahVet\Reef\ClientFactory(
    'https://staging.vetinweb.com'
);
$client = $clientFactory->createClient($bearerToken, \NoahVet\Reef\Bsm\Client::class);
```

### Symfony bundle integration
Import the bundle in you bundles.php :
```php
[
    ...
    NoahVet\Reef\Symfony\ReefApiClientBundle::class => ['all' => true],
]
```

Add envs to your project in the .env.dev file :
```dotenv
REEF_URL=https://
OAUTH_BASE_URL=https://reef-iam.viw.ovh
OAUTH_ID=client_id_here
OAUTH_SECRET=secret_here
REEF_CACHE=service://cache.app
REEF_HTTP_CACHE=array
```

Configure your OAuth client information and reef URL in config/packages/reef.yaml :
```yaml
reef:
  url: '%env(string:REEF_URL)%'
  oauth_base_url: '%env(string:OAUTH_BASE_URL)%'
  client_id: '%env(string:OAUTH_ID)%'
  client_secret: '%env(string:OAUTH_SECRET)%'
  cache_service: '%env(string:REEF_CACHE)%'
  http_cache_service: '%env(string:REEF_HTTP_CACHE)%'
```

### Adding cache support

You can add a cache adapter to speedup token validation, and a cache service to cache http requests.
Example using the standard "app.cache" service from a Symfony project : 
```dotenv
REEF_CACHE=service://cache.app
```

To use an HTTP cache locally, define the http cache url. Example using a redis cache :
```dotenv
REEF_HTTP_CACHE=redis://redis/1
```

See Symfony's documentation to use other cache pool : https://symfony.com/doc/5.4/cache.html


### Adding RabbitMQ support
Add PHP Extensions : pcntl, sockets

Add AMQlib & Symfony process to your project :
```shell
composer require php-amqplib/php-amqplib "^3.5.4"
composer require symfony/process
```

To send a task through RabbitMQ :
```php
class MyService {
    public function __construct(
        private readonly MyCommandAsTask $command,
        private readonly ReefTaskSenderInterface $reefTaskSender,
    ) {
    
    }

    public function sendTask(): void {
        $task = new ConsoleCommandTask(
            $this->command,
            [
                'taskArgument1' => 'value1',
                'taskOption1' => 'value2',            
            ],
        );

        $this->reefTaskSender->sendTask($task);
    }
}
```

To send a notification through RabbitMQ :
```php
class MyService {
    public function __construct(
        private readonly ReefNotificationSenderInterface $reefNotificationSender,
    ) {
    
    }

    public function sendNotification(): void {
        $this->reefNotificationSender
            ->sendNotification(
                Resource::fromString('reef:iam:principal', 'test'),
                'reef:iam:principal:perm:update',
                []
            );
    }
}
```

Notification for a new resource creation :
```php
class MyService {
    public function __construct(
        private readonly ReefNotificationSenderInterface $reefNotificationSender,
    ) {
    
    }

    public function sendNotification(): void {
        $this->reefNotificationSender
            ->sendNotification(
                new ResourceTypeResource('reef:bsmApi:veterinaryEstablishment'),
                'reef:iam:resourceType:perm:create',
                [
                    'id' => 'reef:bsmApi:veterinaryEstablishment:ab923bc1-1917-4a23-9c43-cbaed5d60b2b',
                    // creator is not required
                    'creator' => 'reef:iam:principal:51462dd7-7fca-4210-bde7-90bfbf2b8dbf',                
                ]
            );
    }
}
```

Execute RabbitMQ consumer :
```shell
php bin/console reef:rabbitmq:process
```

To display debug information, add the -vvv option.

## Symfony firewall integration (Interactif login)

### Using HWI OAuth Bundle

Install the bundle on your project
```shell
composer require hwi/oauth-bundle:dev-master
```

Configure the bundle in config/packages/hwi_oauth.yaml :
```yaml
hwi_oauth:
  resource_owners:
    noahvet:
      type: oauth2
      class: NoahVet\Reef\HWI\OAuth\ResourceOwner\ReefResourceOwner
      client_id: "%reef.client_id%"
      client_secret: "%reef.client_secret%"
      base_url: "%reef.oauth_base_url%"
      options:
        refresh_on_expire: true
```

Configure your firewall in security.yaml :
```yaml
providers:
  reef_users:
    id: NoahVet\Reef\HWI\OAuth\ReefUserProvider

firewalls:
  main:
    pattern: ^your_pattern_here
    oauth:
      resource_owners:
        noahvet: "/login/noahvet"
      login_path: /login
      use_forward: false
      failure_path: /login
      oauth_user_provider:
        service: NoahVet\Reef\HWI\OAuth\ReefUserProvider
    provider: reef_users
    logout:
      path: app_logout
```

Add a route in routes.yaml :
```yaml
noahvet_login:
  path: /login/noahvet
```

To customize the login template, create a file named templates/bundles/HWIOauthBundle/Connect/login.html.twig

## Symfony firewall integration (API)

Authentication using a bearer token issued from the IAM :
```yaml
security:
  firewalls:
    api:
      pattern: ^your_pattern_here
      stateless: true
      custom_authenticators:
        - NoahVet\Reef\Security\ReefOAuthAuthenticatorInterface

access_control:
  - { path: '^/login', roles: PUBLIC_ACCESS }
  - { path: '^/connect/', roles: PUBLIC_ACCESS }
```

Get a client
```php
public function __construct(
  ClientFactory $clientFactory,
  TokenStorageInterface $tokenStorage
  ) {
  $this->clientFactory = $clientFactory;
}

public function myAction() {
  // Note : token is a ReefOAuthToken if behind a firewall
  $bearerToken = $this->tokenStorage->getToken()->getBearerToken();
  // Basic user info is preloaded in the the use
  $user = $this->tokenStorage->getToken()->getUser();
  $apiClient = $this->clientFactory->create($bearerToken);
}
```

Check permissions on a resource type or a resource

```php
use NoahVet\Reef\Security\IAM\Model\Resource;use NoahVet\Reef\Security\IAM\Model\ResourceType;

class MyController extends AbstractController {
  public function __invoke(): Response {
    // Check if the current user can create a new resource of the given resource type
    $this->denyAccessUnlessGranted(
      'reef:iam:resourceType:perm:create',
      ResourceType::fromString('reef:bsmApi:myResourceType')
    );
    
    $object = ...
    
    if ( $this->isGranted(
      'reef:myService:resourceType:perm:myPermission',
      Resource::fromString('reef:myService:resourceType', $object->getId())
      ) {
        // Only if the permission is granted...
      }
    );
  }
}
```

Filter a query of a specific resource type

```php
use NoahVet\Reef\Security\IAM\Model\ResourceType;
use NoahVet\Reef\Security\ResourceType\ResourceCollectionPermissionFetcherInterface;

class MyController extends AbstractController {
  public function __invoke(
    ResourceCollectionPermissionFetcherInterface $permissionFetcher
  ): Response {
    $permissions = $permissionFetcher->fetch(
        ResourceType::fromString('reef:bsmApi:myResourceType')
    );    
  
    $queryBuilder = ...
  
    if (in_array('reef:myService:resourceType:perm:myPermission', $permissions->allPermissions)) {
        // User can "myPermission" all resources 
    } else {
        // Filter query to use only allowed ids    
        $queryBuilder
        ->andWhere('id IN (:ids)')
        ->setParameters('ids', $permissions->filterResources(['reef:myService:resourceType:perm:myPermission']));
    }
  
    // Check if the current user can create a new resource of the given resource type
    $this->denyAccessUnlessGranted(
        'reef:iam:resourceType:perm:create',
        ResourceType::fromString('reef:bsmApi:myResourceType')
    );
    
    $objects = $queryBuilder->getQuery()->getResult();
  }
}
```

## Logging

### Audit

Audit logging is done through the event system of Symfony.

In a controller, use the _NoahVet\Reef\Event\RequestAuditEvent_ class
```php
$this->eventDispatcher->dispatch(
    new RequestAuditEvent(
        'reef:myApp:myResourceType:perm:myPermission',
        'reef:myApp:myResourceType:resourceId',
        'reef:myApp:myResourceType',
        'Short description',
        [
            'extraParam' => 'value',            
        ],
    ),
);
```

In a command, uses the _NoahVet\Reef\Event\CommandAuditEvent_ class
```php
$this->eventDispatcher->dispatch(
    new CommandAuditEvent(
        'reef:myApp:myResourceType:perm:myPermission',
        'reef:myApp:myResourceType:resourceId',
        'reef:myApp:myResourceType',
        'Short description',
        [
            'extraParam' => 'value',            
        ],
    ),
);
```

## Support
This client supports these reef features :
 - Authentication using IAM
 - Easy permission checking
 - Audit logging

Platform support :
 - Symfony 5.4
 - Symfony 6.4


## Availables API clients
| Project | Composer package        |
|---------|-------------------------|
| BSM     | noahvet/reef-bsm-client |
| CAM     | noahvet/reef-cam-client |
| IAM     | noahvet/reef-iam-client |
| INS     | noahvet/reef-ins-client |
| PIM     | noahvet/reef-pim-client |

## Roadmap
Ce client est pour le moment un couteau-suisse pour les développements internes.
Il a vocation à être séparé en plusieurs projets, certains internes d'autres pour faciliter le travail des développeurs externes.
