<?php

declare(strict_types=1);

namespace NoahVet\Reef\RabbitMQ\Notification;

use NoahVet\Reef\Security\IAM\Model\Resource;

class ReefNotification implements ReefNotificationInterface
{
    /**
     * Client name of App generating the notification.
     *
     * @example reef:bsmApi
     */
    public readonly string $clientId;

    /**
     * Resource type of the subject.
     *
     * @example reef:bsmApi:veterinaryEstablishment
     */
    public readonly string $resourceType;

    /**
     * Resource id (excluding resourceType Part).
     *
     * @example ec17f420-363d-4b27-ad3d-4d4dd6eeda34
     */
    public readonly string $resourceId;

    /**
     * The action performed on the resource (must be a fully qualified IAM permission slug).
     *
     * @example reef:bsmApi:veterinaryEstablishment:perm:updatePublic
     * @example reef:iam:resourceType:perm:create (Creation of a new resource).
     */
    public readonly string $action;

    /**
     * @var array<string, string|int|float|boolean|array<string, string|int|float|boolean>>
     */
    public readonly array $rawData;

    /**
     * @param array<string, string|int|float|boolean|array<string, string|int|float|boolean>> $rawData
     */
    public function __construct(
        string $clientId,
        string $resourceType,
        string $resourceId,
        string $action,
        array $rawData)
    {
        $this->clientId = $clientId;
        $this->resourceType = $resourceType;
        $this->resourceId = $resourceId;
        $this->action = $action;
        $this->rawData = $rawData;
    }

    public function getClientId(): string
    {
        return $this->clientId;
    }

    public function getResource(): Resource
    {
        return Resource::fromString($this->resourceType, $this->resourceId);
    }

    public function getAction(): string
    {
        return $this->action;
    }

    public function getRawData(): array
    {
        return $this->rawData;
    }
}
