<?php

declare(strict_types=1);

namespace NoahVet\Reef\Jane\Endpoint;

class GetAllRequiredMandate extends \NoahVet\Reef\Jane\Runtime\Client\BaseEndpoint implements \NoahVet\Reef\Jane\Runtime\Client\Endpoint
{
    use \NoahVet\Reef\Jane\Runtime\Client\EndpointTrait;

    /**
     * List Required Mandates.
     * A Required Mandate is a company, enterprise, association identifiable by a legalCode.
     *
     * @param array $queryParameters {
     *
     * @var int    $page the page of items to collect a part of result set
     * @var int    $limit the maximum number of items to collect a part of result set
     * @var array  $id[] Filter by id
     * @var array  $billingAccountId[] Filter by billingAccountId
     * @var array  $mandateId[] Filter by mandateId
     * @var bool   $isValidated Filter by validated / not validated
     * @var array  $fromCreatedDate[] Filter from createdDate
     * @var array  $fromUpdatedDate[] Filter from updatedDate
     * @var array  $toCreatedDate[] Filter to createdDate
     * @var array  $toUpdatedDate[] Filter to updatedDate
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     */
    public function __construct(array $queryParameters = [], array $headerParameters = [])
    {
        $this->queryParameters = $queryParameters;
        $this->headerParameters = $headerParameters;
    }

    public function getMethod(): string
    {
        return 'GET';
    }

    public function getUri(): string
    {
        return '/v1/required_mandates';
    }

    public function getBody(\Symfony\Component\Serializer\SerializerInterface $serializer, $streamFactory = null): array
    {
        return [[], null];
    }

    public function getExtraHeaders(): array
    {
        return ['Accept' => ['application/json']];
    }

    public function getAuthenticationScopes(): array
    {
        return ['BearerAuth'];
    }

    protected function getQueryOptionsResolver(): \Symfony\Component\OptionsResolver\OptionsResolver
    {
        $optionsResolver = parent::getQueryOptionsResolver();
        $optionsResolver->setDefined(['page', 'limit', 'id', 'billingAccountId', 'mandateId', 'isValidated', 'fromCreatedDate', 'fromUpdatedDate', 'toCreatedDate', 'toUpdatedDate', 'sortCreatedDate', 'sortUpdatedDate']);
        $optionsResolver->setRequired([]);
        $optionsResolver->setDefaults([]);
        $optionsResolver->addAllowedTypes('page', ['int']);
        $optionsResolver->addAllowedTypes('limit', ['int']);
        $optionsResolver->addAllowedTypes('id', ['array']);
        $optionsResolver->addAllowedTypes('billingAccountId', ['array']);
        $optionsResolver->addAllowedTypes('mandateId', ['array']);
        $optionsResolver->addAllowedTypes('isValidated', ['bool']);
        $optionsResolver->addAllowedTypes('fromCreatedDate', ['array']);
        $optionsResolver->addAllowedTypes('fromUpdatedDate', ['array', 'null']);
        $optionsResolver->addAllowedTypes('toCreatedDate', ['array', 'null']);
        $optionsResolver->addAllowedTypes('toUpdatedDate', ['array', 'null']);
        $optionsResolver->addAllowedTypes('sortCreatedDate', ['string']);
        $optionsResolver->addAllowedTypes('sortUpdatedDate', ['string']);

        return $optionsResolver;
    }

    protected function getHeadersOptionsResolver(): \Symfony\Component\OptionsResolver\OptionsResolver
    {
        $optionsResolver = parent::getHeadersOptionsResolver();
        $optionsResolver->setDefined(['Accept-Language']);
        $optionsResolver->setRequired([]);
        $optionsResolver->setDefaults([]);
        $optionsResolver->addAllowedTypes('Accept-Language', ['string']);

        return $optionsResolver;
    }

    /**
     * @return \NoahVet\Reef\Jane\Model\RequiredMandateResponseItem[]
     *
     * @throws \NoahVet\Reef\Jane\Exception\GetAllRequiredMandateBadRequestException
     * @throws \NoahVet\Reef\Jane\Exception\GetAllRequiredMandateInternalServerErrorException
     * @throws \NoahVet\Reef\Jane\Exception\UnexpectedStatusCodeException
     */
    protected function transformResponseBody(\Psr\Http\Message\ResponseInterface $response, \Symfony\Component\Serializer\SerializerInterface $serializer, string $contentType = null)
    {
        $status = $response->getStatusCode();
        $body = (string) $response->getBody();
        if ((null === $contentType) === false && (200 === $status && false !== \mb_strpos($contentType, 'application/json'))) {
            return $serializer->deserialize($body, 'NoahVet\\Reef\\Jane\\Model\\RequiredMandateResponseItem[]', 'json');
        }
        if ((null === $contentType) === false && (206 === $status && false !== \mb_strpos($contentType, 'application/json'))) {
            return $serializer->deserialize($body, 'NoahVet\\Reef\\Jane\\Model\\RequiredMandateResponseItem[]', 'json');
        }
        if (400 === $status) {
            throw new \NoahVet\Reef\Jane\Exception\GetAllRequiredMandateBadRequestException($response);
        }
        if (500 === $status) {
            throw new \NoahVet\Reef\Jane\Exception\GetAllRequiredMandateInternalServerErrorException($response);
        }
        throw new \NoahVet\Reef\Jane\Exception\UnexpectedStatusCodeException($status, $body);
    }
}
