<?php

declare(strict_types=1);

namespace NoahVet\Reef\Test\A_Unit\File\Dumper\Yaml;

use NoahVet\Reef\File\Dumper\Yaml\Dumper;
use NoahVet\Reef\File\Dumper\Yaml\DumperInterface;
use PHPUnit\Framework\TestCase;
use Symfony\Component\Yaml\Yaml;

final class DumperTest extends TestCase
{
    private DumperInterface $subject;

    protected function setUp(): void
    {
        parent::setUp();
        $this->subject = new Dumper();
    }

    public function testDumpContentProducesExpectedYamlWithIndentTwo(): void
    {
        $content = [
            'name' => 'App',
            'version' => 1,
            'features' => [
                'a' => true,
                'b' => false,
            ],
            'list' => ['x', 'y'],
        ];

        $expected = Yaml::dump(
            $content,
            32,
            2,
            Yaml::PARSE_CONSTANT,
        );

        $actual = $this->subject->dumpContent($content);

        $this->assertSame($expected, $actual);
        $this->assertStringContainsString("name: App\n", $actual);
        $this->assertStringContainsString("  a: true\n", $actual); // 2-space indent for nested mapping
        $this->assertStringContainsString("  - x\n", $actual);     // 2-space indent for sequence items
    }

    public function testDumpContentWithEmptyArrayReturnsEmptyListNotation(): void
    {
        $expected = Yaml::dump([], 32, 2, Yaml::PARSE_CONSTANT);

        $actual = $this->subject->dumpContent([]);

        $this->assertSame($expected, $actual);
    }

    public function testDumpContentHandlesDeeplyNestedStructuresWithoutInlining(): void
    {
        $content = [
            'root' => [
                'level1' => [
                    'level2' => [
                        'level3' => ['k' => 'v'],
                    ],
                ],
            ],
        ];

        $actual = $this->subject->dumpContent($content);

        // Ensure multi-line block style (not inlined) due to inline depth 32 and 2-space indentation
        $this->assertStringContainsString("root:\n", $actual);
        $this->assertStringContainsString("  level1:\n", $actual);
        $this->assertStringContainsString("    level2:\n", $actual);
        $this->assertStringContainsString("      level3:\n", $actual);
        $this->assertStringContainsString("        k: v\n", $actual);
    }
}
