<?php

declare(strict_types=1);

namespace NoahVet\Reef\Test\A_Unit\Domain\Tool;

use NoahVet\Reef\Domain\Tool\OpenAPITool;
use NoahVet\Reef\Domain\Tool\OpenAPIToolInterface;
use PHPUnit\Framework\TestCase;

final class OpenAPIToolTest extends TestCase
{
    private OpenAPIToolInterface $subject;

    protected function setUp(): void
    {
        parent::setUp();
        $this->subject = new OpenAPITool();
    }

    public function testCleanParametersExtractsQueryNamesAndStripsBrackets(): void
    {
        $content = [
            'parameters' => [
                ['in' => 'query', 'name' => 'page'],
                ['in' => 'header', 'name' => 'X-Api-Key'],
                ['in' => 'query', 'name' => 'filter[status]'],
                ['in' => 'query', 'name' => 'tags[]'],
            ],
        ];

        $result = $this->subject->cleanParameters($content);

        $this->assertSame(
            ['parameters' => ['page', 'filterstatus', 'tags']],
            $result,
        );
    }

    public function testCleanParametersIgnoresNonQueryAndMissingIn(): void
    {
        $content = [
            'parameters' => [
                ['in' => 'path', 'name' => 'id'],
                ['name' => 'noIn'],
                ['in' => 'cookie', 'name' => 'sid'],
            ],
        ];

        $result = $this->subject->cleanParameters($content);

        $this->assertSame([], $result);
    }

    public function testCleanParametersWithEmptyListReturnsEmptyArray(): void
    {
        $result = $this->subject->cleanParameters(['parameters' => []]);

        $this->assertSame([], $result);
    }

    public function testCleanUriRemovesBraces(): void
    {
        $uri = '/pets/{petId}/owners/{ownerId}';

        $result = $this->subject->cleanUri($uri);

        $this->assertSame('/pets/petId/owners/ownerId', $result);
    }

    public function testCleanUriReturnsUnchangedWhenNoBraces(): void
    {
        $uri = '/health';

        $result = $this->subject->cleanUri($uri);

        $this->assertSame('/health', $result);
    }
}
