<?php

declare(strict_types=1);

namespace NoahVet\Reef\Test\A_Unit\Domain\Http;

use NoahVet\Reef\Domain\Http\CacheHeaderParser;
use PHPUnit\Framework\TestCase;
use Psr\Http\Message\ResponseInterface;

class CacheHeaderParserTest extends TestCase
{
    public function testGetExpirationDateNoCache(): void
    {
        $response = $this->createMock(ResponseInterface::class);
        $response->method('getHeaderLine')
            ->willReturnMap([
                ['Cache-Control', 'no-cache'],
                ['Date', ''],
                ['Expires', ''],
            ])
        ;

        $expirationDate = CacheHeaderParser::getExpirationDate($response);

        $this->assertNull($expirationDate);
    }

    public function testGetExpirationDateNoStore(): void
    {
        $response = $this->createMock(ResponseInterface::class);
        $response->method('getHeaderLine')
            ->willReturnMap([
                ['Cache-Control', 'no-store'],
                ['Date', ''],
                ['Expires', ''],
            ])
        ;

        $expirationDate = CacheHeaderParser::getExpirationDate($response);

        $this->assertNull($expirationDate);
    }

    public function testGetExpirationDateWithMaxAge(): void
    {
        $response = $this->createMock(ResponseInterface::class);
        $response->method('getHeaderLine')
            ->willReturnMap([
                ['Cache-Control', 'max-age=3600'],
                ['Date', 'Wed, 21 Oct 2015 07:28:00 GMT'],
                ['Expires', ''],
            ])
        ;

        $expirationDate = CacheHeaderParser::getExpirationDate($response);
        $expectedDate = new \DateTimeImmutable('+3600 seconds');

        $this->assertEquals($expectedDate->format('c'), $expirationDate->format('c'));
    }

    public function testGetExpirationDateExpiresHeader(): void
    {
        $response = $this->createMock(ResponseInterface::class);
        $response->method('getHeaderLine')
            ->willReturnMap([
                ['Cache-Control', ''],
                ['Date', ''],
                ['Expires', 'Wed, 21 Oct 2015 07:28:00 GMT'],
            ])
        ;

        $expectedDate = new \DateTimeImmutable('Wed, 21 Oct 2015 07:28:00 GMT');
        $expirationDate = CacheHeaderParser::getExpirationDate($response);

        $this->assertEquals($expectedDate, $expirationDate);
    }

    public function testGetExpirationDateNoValidHeaders(): void
    {
        $response = $this->createMock(ResponseInterface::class);
        $response->method('getHeaderLine')
            ->willReturnMap([
                ['Cache-Control', ''],
                ['Date', ''],
                ['Expires', ''],
            ])
        ;

        $expirationDate = CacheHeaderParser::getExpirationDate($response);

        $this->assertNull($expirationDate);
    }
}
