<?php

declare(strict_types=1);

namespace NoahVet\Reef\Jane\Endpoint;

class GetAllSupplierOrder extends \NoahVet\Reef\Jane\Runtime\Client\BaseEndpoint implements \NoahVet\Reef\Jane\Runtime\Client\Endpoint
{
    use \NoahVet\Reef\Jane\Runtime\Client\EndpointTrait;

    /**
     * List supplier orders.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $veterinaryEstablishmentId[] Filter by veterinaryEstablishmentId
     * @var array $fromInvoiceDate Returns results whose `invoiceDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toInvoiceDate Returns results whose `invoiceDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortInvoiceDate Sort by invoiceDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     */
    public function __construct(array $queryParameters = [], array $headerParameters = [])
    {
        $this->queryParameters = $queryParameters;
        $this->headerParameters = $headerParameters;
    }

    public function getMethod(): string
    {
        return 'GET';
    }

    public function getUri(): string
    {
        return '/v1/supplier_orders';
    }

    public function getBody(\Symfony\Component\Serializer\SerializerInterface $serializer, $streamFactory = null): array
    {
        return [[], null];
    }

    public function getExtraHeaders(): array
    {
        return ['Accept' => ['application/json']];
    }

    public function getAuthenticationScopes(): array
    {
        return ['BearerAuth'];
    }

    protected function getQueryOptionsResolver(): \Symfony\Component\OptionsResolver\OptionsResolver
    {
        $optionsResolver = parent::getQueryOptionsResolver();
        $optionsResolver->setDefined(['page', 'limit', 'veterinaryEstablishmentId', 'fromInvoiceDate', 'toInvoiceDate', 'sortInvoiceDate']);
        $optionsResolver->setRequired([]);
        $optionsResolver->setDefaults([]);
        $optionsResolver->addAllowedTypes('page', ['int']);
        $optionsResolver->addAllowedTypes('limit', ['int']);
        $optionsResolver->addAllowedTypes('veterinaryEstablishmentId', ['array']);
        $optionsResolver->addAllowedTypes('fromInvoiceDate', ['array']);
        $optionsResolver->addAllowedTypes('toInvoiceDate', ['array', 'null']);
        $optionsResolver->addAllowedTypes('sortInvoiceDate', ['string']);

        return $optionsResolver;
    }

    protected function getHeadersOptionsResolver(): \Symfony\Component\OptionsResolver\OptionsResolver
    {
        $optionsResolver = parent::getHeadersOptionsResolver();
        $optionsResolver->setDefined(['Accept-Language']);
        $optionsResolver->setRequired([]);
        $optionsResolver->setDefaults([]);
        $optionsResolver->addAllowedTypes('Accept-Language', ['string']);

        return $optionsResolver;
    }

    /**
     * @return \NoahVet\Reef\Jane\Model\SupplierOrderResponse[]
     *
     * @throws \NoahVet\Reef\Jane\Exception\GetAllSupplierOrderBadRequestException
     * @throws \NoahVet\Reef\Jane\Exception\UnexpectedStatusCodeException
     */
    protected function transformResponseBody(\Psr\Http\Message\ResponseInterface $response, \Symfony\Component\Serializer\SerializerInterface $serializer, ?string $contentType = null)
    {
        $status = $response->getStatusCode();
        $body = (string) $response->getBody();
        if ((null === $contentType) === false && (200 === $status && false !== \mb_strpos($contentType, 'application/json'))) {
            return $serializer->deserialize($body, 'NoahVet\\Reef\\Jane\\Model\\SupplierOrderResponse[]', 'json');
        }
        if ((null === $contentType) === false && (206 === $status && false !== \mb_strpos($contentType, 'application/json'))) {
            return $serializer->deserialize($body, 'NoahVet\\Reef\\Jane\\Model\\SupplierOrderResponse[]', 'json');
        }
        if (400 === $status) {
            throw new \NoahVet\Reef\Jane\Exception\GetAllSupplierOrderBadRequestException($response);
        }
        throw new \NoahVet\Reef\Jane\Exception\UnexpectedStatusCodeException($status, $body);
    }
}
