<?php

declare(strict_types=1);

namespace NoahVet\Reef\Jane;

class Client extends Runtime\Client\Client
{
    /**
     * Get A.
     *
     * @param string $a     the dynamic name
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\GetABadRequestException
     * @throws Exception\GetANotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getA(string $a, string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetA($a), $fetch);
    }

    /**
     * Get AB.
     *
     * @param string $a     the dynamic name
     * @param string $b     the dynamic name
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\GetABBadRequestException
     * @throws Exception\GetABNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAB(string $a, string $b, string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAB($a, $b), $fetch);
    }

    /**
     * Get ABC.
     *
     * @param string $a     the dynamic name
     * @param string $b     the dynamic name
     * @param string $c     the dynamic name
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\GetABCBadRequestException
     * @throws Exception\GetABCNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getABC(string $a, string $b, string $c, string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetABC($a, $b, $c), $fetch);
    }

    /**
     * Get ABCD.
     *
     * @param string $a     the dynamic name
     * @param string $b     the dynamic name
     * @param string $c     the dynamic name
     * @param string $d     the dynamic name
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\GetABCDBadRequestException
     * @throws Exception\GetABCDNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getABCD(string $a, string $b, string $c, string $d, string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetABCD($a, $b, $c, $d), $fetch);
    }

    /**
     * Get ABCDE.
     *
     * @param string $a     the dynamic name
     * @param string $b     the dynamic name
     * @param string $c     the dynamic name
     * @param string $d     the dynamic name
     * @param string $e     the dynamic name
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\GetABCDEBadRequestException
     * @throws Exception\GetABCDENotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getABCDE(string $a, string $b, string $c, string $d, string $e, string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetABCDE($a, $b, $c, $d, $e), $fetch);
    }

    /**
     * Get Tmp Contract Document.
     *
     * @param string $a     the dynamic name
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\GetTmpContractDocumentBadRequestException
     * @throws Exception\GetTmpContractDocumentNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getTmpContractDocument(string $a, string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetTmpContractDocument($a), $fetch);
    }

    /**
     * Get Tmp Contract Document Signed.
     *
     * @param string $a     the dynamic name
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\GetTmpContractDocumentSignedBadRequestException
     * @throws Exception\GetTmpContractDocumentSignedNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getTmpContractDocumentSigned(string $a, string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetTmpContractDocumentSigned($a), $fetch);
    }

    /**
     * Get Tmp Required Mandate.
     *
     * @param string $a     the dynamic name
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\GetTmpRequiredMandateBadRequestException
     * @throws Exception\GetTmpRequiredMandateNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getTmpRequiredMandate(string $a, string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetTmpRequiredMandate($a), $fetch);
    }

    /**
     * Get Tmp Veterinary Establishment Credit Note.
     *
     * @param string $a     the dynamic name
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\GetTmpVeterinaryEstablishmentCreditNoteBadRequestException
     * @throws Exception\GetTmpVeterinaryEstablishmentCreditNoteNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getTmpVeterinaryEstablishmentCreditNote(string $a, string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetTmpVeterinaryEstablishmentCreditNote($a), $fetch);
    }

    /**
     * Get Tmp Veterinary Establishment Free Credit Note.
     *
     * @param string $a     the dynamic name
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\GetTmpVeterinaryEstablishmentFreeCreditNoteBadRequestException
     * @throws Exception\GetTmpVeterinaryEstablishmentFreeCreditNoteNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getTmpVeterinaryEstablishmentFreeCreditNote(string $a, string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetTmpVeterinaryEstablishmentFreeCreditNote($a), $fetch);
    }

    /**
     * Get Tmp Veterinary Establishment Free Invoice.
     *
     * @param string $a     the dynamic name
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\GetTmpVeterinaryEstablishmentFreeInvoiceBadRequestException
     * @throws Exception\GetTmpVeterinaryEstablishmentFreeInvoiceNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getTmpVeterinaryEstablishmentFreeInvoice(string $a, string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetTmpVeterinaryEstablishmentFreeInvoice($a), $fetch);
    }

    /**
     * Get Tmp Veterinary Establishment Invoice.
     *
     * @param string $a     the dynamic name
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\GetTmpVeterinaryEstablishmentInvoiceBadRequestException
     * @throws Exception\GetTmpVeterinaryEstablishmentInvoiceNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getTmpVeterinaryEstablishmentInvoice(string $a, string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetTmpVeterinaryEstablishmentInvoice($a), $fetch);
    }

    /**
     * Get Tmp Veterinary Group Invoice.
     *
     * @param string $a     the dynamic name
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\GetTmpVeterinaryGroupInvoiceBadRequestException
     * @throws Exception\GetTmpVeterinaryGroupInvoiceNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getTmpVeterinaryGroupInvoice(string $a, string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetTmpVeterinaryGroupInvoice($a), $fetch);
    }

    /**
     * List Billing Accounts.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $legalStructureId[] Filter by legalStructureId
     * @var array $status[]
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\BillingAccountResponseItem[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllBillingAccountBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllBillingAccount(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllBillingAccount($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a Billing Account.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\SchemasBillingAccountResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostBillingAccountBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postBillingAccount(Model\BillingAccountRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostBillingAccount($requestBody, $headerParameters), $fetch);
    }

    /**
     * Delete a Billing Account.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\DeleteOneBillingAccountBadRequestException
     * @throws Exception\DeleteOneBillingAccountNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function deleteOneBillingAccount(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\DeleteOneBillingAccount($id, $headerParameters), $fetch);
    }

    /**
     * Get a Billing Account.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ComponentsSchemasBillingAccountResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetOneBillingAccountBadRequestException
     * @throws Exception\GetOneBillingAccountNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getOneBillingAccount(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetOneBillingAccount($id, $headerParameters), $fetch);
    }

    /**
     * Update a Billing Account.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\BillingAccountResponseComponentsSchemasBillingAccountResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneBillingAccountBadRequestException
     * @throws Exception\UpdateOneBillingAccountNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneBillingAccount(string $id, Model\SchemasBillingAccountRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneBillingAccount($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * Get an image preview of Required Mandate by Billing Account Id.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept the request accept language for content negotiation
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\GetOneImagePreviewRequiredMandateBadRequestException
     * @throws Exception\GetOneImagePreviewRequiredMandateNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getOneImagePreviewRequiredMandate(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetOneImagePreviewRequiredMandate($id, $headerParameters), $fetch);
    }

    /**
     * Get a Yousign Signature Request by Legal Structure Id.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept the request accept language for content negotiation
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\YousignSignatureRequestResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetOneYousignSignatureRequestBadRequestException
     * @throws Exception\GetOneYousignSignatureRequestNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getOneYousignSignatureRequest(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetOneYousignSignatureRequest($id, $headerParameters), $fetch);
    }

    /**
     * Create an CreditNoteBankTransfer.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\CreditNoteBankTransferResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostCreditNoteBankTransferBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postCreditNoteBankTransfer(Model\CreditNoteBankTransferRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostCreditNoteBankTransfer($requestBody, $headerParameters), $fetch);
    }

    /**
     * Create an InvoiceDirectDebit.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\InvoiceDirectDebitResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostInvoiceDirectDebitBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postInvoiceDirectDebit(Model\InvoiceDirectDebitRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostInvoiceDirectDebit($requestBody, $headerParameters), $fetch);
    }

    /**
     * List Mandates.
     * A Mandate is a company, enterprise, association identifiable by a legalCode.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $prefix[] Filter by prefix
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\MandateResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllMandateBadRequestException
     * @throws Exception\GetAllMandateInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllMandate(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllMandate($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Get a Mandate.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept the request accept language for content negotiation
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\MandateResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetOneMandateBadRequestException
     * @throws Exception\GetOneMandateNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getOneMandate(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetOneMandate($id, $headerParameters), $fetch);
    }

    /**
     * Confirm a Yousign Signature Request.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept the request accept language for content negotiation
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\GetOneYousignSignatureRequestConfirmationBadRequestException
     * @throws Exception\GetOneYousignSignatureRequestConfirmationNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getOneYousignSignatureRequestConfirmation(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetOneYousignSignatureRequestConfirmation($id, $headerParameters), $fetch);
    }

    /**
     * List Contracts.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ContractResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllContractBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllContract(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllContract($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a Contract.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ContractResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostContractBadRequestException
     * @throws Exception\PostContractGoneException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postContract(Model\ContractRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostContract($requestBody, $headerParameters), $fetch);
    }

    /**
     * Get a Contract YouSign Signature Request.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept the request accept language for content negotiation
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\GetOneContractYouSignSignatureRequestBadRequestException
     * @throws Exception\GetOneContractYouSignSignatureRequestNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getOneContractYouSignSignatureRequest(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetOneContractYouSignSignatureRequest($id, $headerParameters), $fetch);
    }

    /**
     * List Contract Documents.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $status[]
     * @var array $id[] Filter by id
     * @var array $contractId[] Filter by contractId
     * @var array $signerId[] Filter by signer id
     * @var array $veterinaryEstablishmentId[] Filter by veterinaryEstablishmentId
     * @var array $veterinaryEstablishmentName[] Filter by veterinaryEstablishmentName
     * @var array $veterinaryEstablishmentPrivilegedCode[] Filter by veterinaryEstablishmentPrivilegedCode
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortId sort by id
     * @var string $sortSignerId sort by signer id
     * @var string $sortStatus sort by status
     * @var string $sortUpdatedDate sort by updatedDate
     * @var string $sortVeterinaryEstablishmentName sort by veterinaryEstablishmentName
     * @var string $sortVeterinaryEstablishmentPrivilegedCode Sort by veterinaryEstablishmentPrivilegedCode.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ContractDocumentResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllContractDocumentBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllContractDocument(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllContractDocument($queryParameters, $headerParameters), $fetch);
    }

    /**
     * List Invitations.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $hostUserId[] Filter by hostUserId
     * @var array $invitedUserId[] Filter by invitedUserId
     * @var array $status[] Filter by status
     * @var array $type[] Filter by type
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromExpirationDate[] Returns results whose `expirationDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toExpirationDate[] Returns results whose `expirationDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate sort by updatedDate
     * @var string $sortExpirationDate Sort by expirationDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\InvitationResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllInvitationBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllInvitation(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllInvitation($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Send a new email invitation.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept the request accept language for content negotiation
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\RemindInvitationNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function remindInvitation(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\RemindInvitation($id, $headerParameters), $fetch);
    }

    /**
     * Create a Veterinary Establishment User Invitation.
     * A Veterinary Establishment User Invitations is an invitation to join a veterinary establishment.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentUserInvitationResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostVeterinaryEstablishmentUserInvitationBadRequestException
     * @throws Exception\PostVeterinaryEstablishmentUserInvitationConflictException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postVeterinaryEstablishmentUserInvitation(Model\VeterinaryEstablishmentUserInvitationRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostVeterinaryEstablishmentUserInvitation($requestBody, $headerParameters), $fetch);
    }

    /**
     * Create a Veterinary Establishment User Invitation Confirmation.
     * A Veterinary Establishment User Invitation Confirmation is an confirmation to join or not a veterinary establishment.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentUserInvitationConfirmationResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostVeterinaryEstablishmentUserInvitationConfirmationBadRequestException
     * @throws Exception\PostVeterinaryEstablishmentUserInvitationConfirmationNotFoundException
     * @throws Exception\PostVeterinaryEstablishmentUserInvitationConfirmationGoneException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postVeterinaryEstablishmentUserInvitationConfirmation(Model\VeterinaryEstablishmentUserInvitationConfirmationRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostVeterinaryEstablishmentUserInvitationConfirmation($requestBody, $headerParameters), $fetch);
    }

    /**
     * Create a Veterinary Group User Invitation.
     * A Veterinary Group User Invitations is an invitation to join a veterinary group.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryGroupUserInvitationResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostVeterinaryGroupUserInvitationBadRequestException
     * @throws Exception\PostVeterinaryGroupUserInvitationConflictException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postVeterinaryGroupUserInvitation(Model\VeterinaryGroupUserInvitationRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostVeterinaryGroupUserInvitation($requestBody, $headerParameters), $fetch);
    }

    /**
     * Create a Veterinary Group User Invitation Confirmation.
     * A Veterinary Group User Invitation Confirmation is an confirmation to join or not a veterinary group.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryGroupUserInvitationConfirmationResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostVeterinaryGroupUserInvitationConfirmationBadRequestException
     * @throws Exception\PostVeterinaryGroupUserInvitationConfirmationNotFoundException
     * @throws Exception\PostVeterinaryGroupUserInvitationConfirmationGoneException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postVeterinaryGroupUserInvitationConfirmation(Model\VeterinaryGroupUserInvitationConfirmationRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostVeterinaryGroupUserInvitationConfirmation($requestBody, $headerParameters), $fetch);
    }

    /**
     * List Credit Notes.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $category[] Filter by credit note category
     * @var array $paymentMethod[] Filter by paymentMethod
     * @var array $paymentStatus[] Filter by paymentStatus
     * @var array $accountingCode[] Filter by accountingCode
     * @var array $accountingMovementIncrementalNumber[] Filter by accountingMovementIncrementalNumber
     * @var array $bankAccountRum[] Filter by bankAccountRum
     * @var array $deliveryInformationCode[] Filter by deliveryInformationCode
     * @var array $senderAddressIdentityName[] Filter by sender identityName
     * @var array $fromAccountingMovementIncrementalNumber[] Returns results whose `incrementalNumber of accountingMovement` is greater than or equal to the specified value.
     *            Example for FAC-V000002 value :
     *
     * |  FAC-V000001 |  FAC-V000002  FAC-V000003  |
     * |----Filtered--|----------Returned----------|
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toAccountingMovementIncrementalNumber[] Returns results whose `incrementalNumber of accountingMovement` is lower than or equal to the specified value.
     *            Example for FAC-V000002 value :
     *
     * |  FAC-V000001  FAC-V000002 |  FAC-V000003 |
     * |----------Returned---------|---Filtered---|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortAccountingMovementIncrementalNumber sort by incrementalNumber of accountingMovement
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\CreditNoteResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllCreditNoteBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllCreditNote(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllCreditNote($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a CreditNote.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\CreditNoteResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostCreditNoteBadRequestException
     * @throws Exception\PostCreditNoteGoneException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postCreditNote(Model\CreditNoteRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostCreditNote($requestBody, $headerParameters), $fetch);
    }

    /**
     * Update a Credit Note.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\CreditNoteResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneCreditNoteBadRequestException
     * @throws Exception\UpdateOneCreditNoteNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneCreditNote(string $id, Model\SchemasCreditNoteRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneCreditNote($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * List Invoices.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $veterinaryEstablishmentServiceDeliveryId[] Filter by veterinaryEstablishmentServiceDeliveryId
     * @var array $veterinaryEstablishmentServicePeriodId[] Filter by veterinaryEstablishmentServicePeriodId
     * @var array $category[] Filter by invoice category
     * @var array $paymentMethod[] Filter by paymentMethod
     * @var array $paymentStatus[] Filter by paymentStatus
     * @var array $accountingCode[] Filter by accountingCode
     * @var array $accountingMovementIncrementalNumber[] Filter by accountingMovementIncrementalNumber
     * @var array $bankAccountRum[] Filter by bankAccountRum
     * @var array $deliveryInformationCode[] Filter by deliveryInformationCode
     * @var array $senderAddressIdentityName[] Filter by sender identityName
     * @var array $fromAccountingMovementIncrementalNumber[] Returns results whose `incrementalNumber of accountingMovement` is greater than or equal to the specified value.
     *            Example for FAC-V000002 value :
     *
     * |  FAC-V000001 |  FAC-V000002  FAC-V000003  |
     * |----Filtered--|----------Returned----------|
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toAccountingMovementIncrementalNumber[] Returns results whose `incrementalNumber of accountingMovement` is lower than or equal to the specified value.
     *            Example for FAC-V000002 value :
     *
     * |  FAC-V000001  FAC-V000002 |  FAC-V000003 |
     * |----------Returned---------|---Filtered---|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortAccountingMovementIncrementalNumber sort by incrementalNumber of accountingMovement
     * @var string $sortAmountWithoutVat sort by amountWithoutVat
     * @var string $sortAmountWithVat sort by amountWithVat
     * @var string $sortBankAccountRum sort by BankAccountRum
     * @var string $sortCategory sort by category
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortDeliveryInformationCode sort by deliveryInformationCode
     * @var string $sortPaymentMethod sort by paymentStatus
     * @var string $sortPaymentStatus sort by paymentStatus
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\InvoiceResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllInvoiceBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllInvoice(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllInvoice($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Update a Invoice.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\InvoiceResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneInvoiceBadRequestException
     * @throws Exception\UpdateOneInvoiceNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneInvoice(string $id, Model\InvoiceRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneInvoice($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * Create a ServiceDeliveryInvoice.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ServiceDeliveryInvoiceResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostServiceDeliveryInvoiceBadRequestException
     * @throws Exception\PostServiceDeliveryInvoiceGoneException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postServiceDeliveryInvoice(Model\ServiceDeliveryInvoiceRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostServiceDeliveryInvoice($requestBody, $headerParameters), $fetch);
    }

    /**
     * Create a Veterinary Establishment Credit Note.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\CreditNoteResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostVeterinaryEstablishmentCreditNoteBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postVeterinaryEstablishmentCreditNote(Model\VeterinaryEstablishmentCreditNoteRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostVeterinaryEstablishmentCreditNote($requestBody, $headerParameters), $fetch);
    }

    /**
     * Create a Veterinary Establishment Invoice.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\InvoiceResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostVeterinaryEstablishmentInvoiceBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postVeterinaryEstablishmentInvoice(Model\VeterinaryEstablishmentInvoiceRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostVeterinaryEstablishmentInvoice($requestBody, $headerParameters), $fetch);
    }

    /**
     * List Legal Structures.
     * A legal structure is a company, enterprise, association identifiable by a legalCode.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $veterinaryEstablishmentId[] Filter by veterinaryEstablishmentId
     * @var array $veterinaryEstablishmentMemberId[] Filter by veterinaryEstablishmentMemberId
     * @var array $veterinaryGroupId[] Filter by veterinaryGroupId
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\LegalStructureResponseItem[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllLegalStructureBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllLegalStructure(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllLegalStructure($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a Legal Structure.
     *
     * A legal structure is a company, enterprise, association identifiable by a legalCode.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\SchemasLegalStructureResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostLegalStructureBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postLegalStructure(Model\LegalStructureRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostLegalStructure($requestBody, $headerParameters), $fetch);
    }

    /**
     * Get a Legal Structure.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ComponentsSchemasLegalStructureResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetOneLegalStructureBadRequestException
     * @throws Exception\GetOneLegalStructureNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getOneLegalStructure(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetOneLegalStructure($id, $headerParameters), $fetch);
    }

    /**
     * Update a Legal Structure.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\LegalStructureResponseComponentsSchemasLegalStructureResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneLegalStructureBadRequestException
     * @throws Exception\UpdateOneLegalStructureNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneLegalStructure(string $id, Model\SchemasLegalStructureRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneLegalStructure($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * Search on Legal Structures.
     * A legal structure is a company, enterprise, association identifiable by a legalCode.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $veterinaryEstablishmentId[] Filter by veterinaryEstablishmentId
     * @var array $veterinaryEstablishmentMemberId[] Filter by veterinaryEstablishmentMemberId
     * @var array $veterinaryGroupId[] Filter by veterinaryGroupId
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\LegalStructureSearchResponseItem[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\SearchLegalStructureBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function searchLegalStructure(Model\SearchRequest $requestBody, array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\SearchLegalStructure($requestBody, $queryParameters, $headerParameters), $fetch);
    }

    /**
     * List Required Mandates.
     * A Required Mandate is a company, enterprise, association identifiable by a legalCode.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $billingAccountId[] Filter by billingAccountId
     * @var array $mandateId[] Filter by mandateId
     * @var array $rum[] Filter by rum
     * @var bool  $isValidated Filter by validated / not validated
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\RequiredMandateResponseItem[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllRequiredMandateBadRequestException
     * @throws Exception\GetAllRequiredMandateInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllRequiredMandate(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllRequiredMandate($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a Required Mandate.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\SchemasRequiredMandateResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostRequiredMandateBadRequestException
     * @throws Exception\PostRequiredMandateInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postRequiredMandate(Model\RequiredMandateRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostRequiredMandate($requestBody, $headerParameters), $fetch);
    }

    /**
     * Delete a Required Mandate.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\DeleteOneRequiredMandateBadRequestException
     * @throws Exception\DeleteOneRequiredMandateNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function deleteOneRequiredMandate(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\DeleteOneRequiredMandate($id, $headerParameters), $fetch);
    }

    /**
     * Get a Required Mandate.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ComponentsSchemasRequiredMandateResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetOneRequiredMandateBadRequestException
     * @throws Exception\GetOneRequiredMandateNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getOneRequiredMandate(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetOneRequiredMandate($id, $headerParameters), $fetch);
    }

    /**
     * List Manufacturers.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $code[] Filter by code
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ManufacturerResponseItem[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllManufacturerBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllManufacturer(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllManufacturer($queryParameters, $headerParameters), $fetch);
    }

    /**
     * List Documents.
     *
     * @param array $queryParameters {
     *
     * @var int    $page the page of items to collect a part of result set
     * @var int    $limit the maximum number of items to collect a part of result set
     * @var array  $id[] Filter by id
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\DocumentResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllDocumentsBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllDocuments(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllDocuments($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a Document. A Document represents pdf file.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\DocumentResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostDocumentBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postDocument(Model\DocumentRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostDocument($requestBody, $headerParameters), $fetch);
    }

    /**
     * List Images.
     *
     * @param array $queryParameters {
     *
     * @var int    $page the page of items to collect a part of result set
     * @var int    $limit the maximum number of items to collect a part of result set
     * @var array  $id[] Filter by id
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ImageResponseItem[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllImagesBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllImages(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllImages($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a Image. A Image represents gif, jpeg or png file.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\SchemasImageResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostImageBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postImage(Model\ImageRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostImage($requestBody, $headerParameters), $fetch);
    }

    /**
     * Export Credit Note Sales Journals.
     *
     * @param array $queryParameters {
     *
     * @var string $fromCreditNoteId
     * @var string $toCreditNoteId
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\ExportCreditNoteSalesJournalBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function exportCreditNoteSalesJournal(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\ExportCreditNoteSalesJournal($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Export Invoice Sales Journals.
     *
     * @param array $queryParameters {
     *
     * @var string $fromInvoiceId
     * @var string $toInvoiceId
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\ExportInvoiceSalesJournalBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function exportInvoiceSalesJournal(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\ExportInvoiceSalesJournal($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Export Self Billing Sales Journals.
     *
     * @param array $queryParameters {
     *
     * @var string $fromCreditNoteId
     * @var string $toCreditNoteId
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\ExportSelfBillingSalesJournalBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function exportSelfBillingSalesJournal(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\ExportSelfBillingSalesJournal($queryParameters, $headerParameters), $fetch);
    }

    /**
     * List Veterinary Establishment Service Deliveries.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $veterinaryServiceId[] Filter by veterinaryServiceId
     * @var array $veterinaryEstablishmentId[] Filter by veterinaryEstablishmentId
     * @var array $serviceDeliveryOptionType[] Filter by serviceDeliveryOptionType
     * @var bool  $shouldGenerateInvoice Filter by shouldGenerateInvoice
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortAccountingCode sort by AccountingCode
     * @var string $sortCompletionDate sort by completionDate
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortInvoiceDate sort by invoiceDate
     * @var string $sortUnitPriceWithoutVat sort by unitPriceWithoutVat
     * @var string $sortUnitPriceWithVat sort by unitPriceWithVat
     * @var string $sortUpdatedDate sort by updatedDate
     * @var string $sortVeterinaryServiceEntitled Sort by veterinaryServiceEntitled.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentServiceDeliveryResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllVeterinaryEstablishmentServiceDeliveryBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllVeterinaryEstablishmentServiceDelivery(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllVeterinaryEstablishmentServiceDelivery($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a VeterinaryEstablishmentServiceDelivery.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentServiceDeliveryResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostVeterinaryEstablishmentServiceDeliveryBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postVeterinaryEstablishmentServiceDelivery(Model\VeterinaryEstablishmentServiceDeliveryRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostVeterinaryEstablishmentServiceDelivery($requestBody, $headerParameters), $fetch);
    }

    /**
     * Delete a Veterinary Establishment Service Delivery.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\DeleteOneVeterinaryEstablishmentServiceDeliveryBadRequestException
     * @throws Exception\DeleteOneVeterinaryEstablishmentServiceDeliveryNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function deleteOneVeterinaryEstablishmentServiceDelivery(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\DeleteOneVeterinaryEstablishmentServiceDelivery($id, $headerParameters), $fetch);
    }

    /**
     * Update a Veterinary Establishment Service Delivery.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentServiceDeliveryResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneVeterinaryEstablishmentServiceDeliveryBadRequestException
     * @throws Exception\UpdateOneVeterinaryEstablishmentServiceDeliveryNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneVeterinaryEstablishmentServiceDelivery(string $id, Model\SchemasVeterinaryEstablishmentServiceDeliveryRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneVeterinaryEstablishmentServiceDelivery($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * List Service Delivery Options.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $veterinaryServiceId[] Filter by veterinaryServiceId
     * @var array $type[] Filter by type
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate sort by updatedDate
     * @var string $sortAccountingCode sort by AccountingCode
     * @var string $sortEntitled sort by Entitled
     * @var string $sortType sort by type
     * @var string $sortUnitPriceWithoutVat sort by unitPriceWithoutVat
     * @var string $sortVeterinaryServiceEntitled Sort by veterinaryServiceEntitled.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ServiceDeliveryOptionResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllServiceDeliveryOptionBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllServiceDeliveryOption(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllServiceDeliveryOption($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a ServiceDeliveryOption.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ServiceDeliveryOptionResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostServiceDeliveryOptionBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postServiceDeliveryOption(Model\ServiceDeliveryOptionRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostServiceDeliveryOption($requestBody, $headerParameters), $fetch);
    }

    /**
     * Delete a Service Delivery Option.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\DeleteOneServiceDeliveryOptionBadRequestException
     * @throws Exception\DeleteOneServiceDeliveryOptionNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function deleteOneServiceDeliveryOption(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\DeleteOneServiceDeliveryOption($id, $headerParameters), $fetch);
    }

    /**
     * Update a Service Delivery Option.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ServiceDeliveryOptionResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneServiceDeliveryOptionBadRequestException
     * @throws Exception\UpdateOneServiceDeliveryOptionNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneServiceDeliveryOption(string $id, Model\SchemasServiceDeliveryOptionRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneServiceDeliveryOption($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * List Manufacturer Users.
     * A Manufacturer User s a member of the Manufacturer.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $department[] Filter by veterinary establishment department
     * @var array $jobTitle[] Filter by jobTitle
     * @var array $userId[] Filter by userId
     * @var array $manufacturerId[] Filter by manufacturerId
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ManufacturerUserResponseItem[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllManufacturerUserBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllManufacturerUser(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllManufacturerUser($queryParameters, $headerParameters), $fetch);
    }

    /**
     * List Veterinary Establishment Users.
     * A Veterinary Establishment User s a member of the Veterinary Establishment.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $department[] Filter by veterinary establishment department
     * @var array $jobTitle[] Filter by jobTitle
     * @var array $userId[] Filter by userId
     * @var array $veterinaryEstablishmentId[] Filter by veterinaryEstablishmentId
     * @var array $veterinaryServiceOptionId[] Filter by veterinaryServiceOptionId
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromEntryDate[] Returns results whose `entryDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromExitDate[] Returns results whose `exitDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromServicePeriodStartDate[] Returns results whose `servicePeriodStartDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromServicePeriodEndDate[] Returns results whose `servicePeriodEndDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toEntryDate[] Returns results whose `entryDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toExitDate[] Returns results whose `exitDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toServicePeriodStartDate[] Returns results whose `servicePeriodStartDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toServicePeriodEndDate[] Returns results whose `servicePeriodEndDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate sort by updatedDate
     * @var string $sortEntryDate sort by entryDate
     * @var string $sortExitDate sort by exitDate
     * @var string $sortJobTitle Sort by jobTitle.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentUserResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllVeterinaryEstablishmentUserBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllVeterinaryEstablishmentUser(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllVeterinaryEstablishmentUser($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a Veterinary Establishment User.
     * A Veterinary Establishment User is a member of the Veterinary Establishment.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentUserResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostVeterinaryEstablishmentUserBadRequestException
     * @throws Exception\PostVeterinaryEstablishmentUserConflictException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postVeterinaryEstablishmentUser(Model\VeterinaryEstablishmentUserRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostVeterinaryEstablishmentUser($requestBody, $headerParameters), $fetch);
    }

    /**
     * Delete a Veterinary Establishment User.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\DeleteOneVeterinaryEstablishmentUserBadRequestException
     * @throws Exception\DeleteOneVeterinaryEstablishmentUserNotFoundException
     * @throws Exception\DeleteOneVeterinaryEstablishmentUserGoneException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function deleteOneVeterinaryEstablishmentUser(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\DeleteOneVeterinaryEstablishmentUser($id, $headerParameters), $fetch);
    }

    /**
     * Update a Veterinary Establishment User.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ComponentsSchemasVeterinaryEstablishmentUserResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneVeterinaryEstablishmentUserBadRequestException
     * @throws Exception\UpdateOneVeterinaryEstablishmentUserNotFoundException
     * @throws Exception\UpdateOneVeterinaryEstablishmentUserGoneException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneVeterinaryEstablishmentUser(string $id, Model\SchemasVeterinaryEstablishmentUserRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneVeterinaryEstablishmentUser($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * List Veterinary Group Users.
     * A Veterinary Group User s a member of the Veterinary Group.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $department[] Filter by veterinary group department
     * @var array $jobTitle[] Filter by jobTitle
     * @var array $userId[] Filter by userId
     * @var array $veterinaryGroupId[] Filter by veterinaryGroupId
     * @var array $veterinaryServiceOptionId[] Filter by veterinaryServiceOptionId
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromEntryDate[] Returns results whose `entryDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromExitDate[] Returns results whose `exitDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toEntryDate[] Returns results whose `entryDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toExitDate[] Returns results whose `exitDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate sort by updatedDate
     * @var string $sortEntryDate sort by entryDate
     * @var string $sortExitDate sort by exitDate
     * @var string $sortJobTitle Sort by jobTitle.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryGroupUserResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllVeterinaryGroupUserBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllVeterinaryGroupUser(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllVeterinaryGroupUser($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a Veterinary Group User.
     * A Veterinary Group User s a member of the Veterinary Group.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryGroupUserResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostVeterinaryGroupUserBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postVeterinaryGroupUser(Model\VeterinaryGroupUserRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostVeterinaryGroupUser($requestBody, $headerParameters), $fetch);
    }

    /**
     * Delete a Veterinary Group User.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\DeleteOneVeterinaryGroupUserBadRequestException
     * @throws Exception\DeleteOneVeterinaryGroupUserNotFoundException
     * @throws Exception\DeleteOneVeterinaryGroupUserGoneException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function deleteOneVeterinaryGroupUser(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\DeleteOneVeterinaryGroupUser($id, $headerParameters), $fetch);
    }

    /**
     * Update a Veterinary Group User.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ComponentsSchemasVeterinaryGroupUserResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneVeterinaryGroupUserBadRequestException
     * @throws Exception\UpdateOneVeterinaryGroupUserNotFoundException
     * @throws Exception\UpdateOneVeterinaryGroupUserGoneException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneVeterinaryGroupUser(string $id, Model\SchemasVeterinaryGroupUserRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneVeterinaryGroupUser($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * List Veterinary Establishments.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $legalStructureId[] Filter by legalStructureId
     * @var array $veuUserId[] Filter by userId of Veterinary Establishment User
     * @var array $veterinaryEstablishmentMemberId[] Filter by veterinaryEstablishmentMemberId
     * @var array $name[] Filter by name
     * @var array $privilegedCode[] Filter by privilegedCode
     * @var array $type[] Filter by type
     * @var array $veterinaryEstablishmentTag[] Filter by veterinaryEstablishmentTag
     * @var array $veterinaryEstablishmentServicePeriodDate[] Returns results whose `veterinaryEstablishmentServicePeriodDate` is equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * 2021                              2022
     * |----Dec----||----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |--Filtered--|-------------------------Returned------------------------|
     * @var array $veterinaryServiceOptionCode[]
     * @var array $veuJobTitle[] Filter by jobTitle of veterinary establishment user
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortEntryNumber sort by entryNumber
     * @var string $sortName sort by name
     * @var string $sortPrivilegedCode sort by privilegedCode
     * @var string $sortType sort by type
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllVeterinaryEstablishmentBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllVeterinaryEstablishment(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllVeterinaryEstablishment($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a Veterinary Establishment.
     * This endpoint will also create a Veterinary Establishment.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postVeterinaryEstablishment(Model\VeterinaryEstablishmentRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostVeterinaryEstablishment($requestBody, $headerParameters), $fetch);
    }

    /**
     * Export Veterinary Establishments.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\ExportVeterinaryEstablishmentBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function exportVeterinaryEstablishment(array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\ExportVeterinaryEstablishment($headerParameters), $fetch);
    }

    /**
     * Get a Veterinary Establishment.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetOneVeterinaryEstablishmentBadRequestException
     * @throws Exception\GetOneVeterinaryEstablishmentNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getOneVeterinaryEstablishment(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetOneVeterinaryEstablishment($id, $headerParameters), $fetch);
    }

    /**
     * Update a Veterinary Establishment.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneVeterinaryEstablishmentBadRequestException
     * @throws Exception\UpdateOneVeterinaryEstablishmentNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneVeterinaryEstablishment(string $id, Model\SchemasVeterinaryEstablishmentRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneVeterinaryEstablishment($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * Search on Veterinary Establishments.
     * A veterinary establishment is a company, enterprise, association identifiable by a legalCode.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $legalStructureId[] Filter by legalStructureId
     * @var array $veuUserId[] Filter by userId of Veterinary Establishment User
     * @var array $veterinaryEstablishmentMemberId[] Filter by veterinaryEstablishmentMemberId
     * @var array $name[] Filter by name
     * @var array $privilegedCode[] Filter by privilegedCode
     * @var array $type[] Filter by type
     * @var array $veterinaryEstablishmentTag[] Filter by veterinaryEstablishmentTag
     * @var array $veterinaryEstablishmentServicePeriodDate[] Returns results whose `veterinaryEstablishmentServicePeriodDate` is equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * 2021                              2022
     * |----Dec----||----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |--Filtered--|-------------------------Returned------------------------|
     * @var array $veterinaryServiceOptionCode[]
     * @var array $veuJobTitle[] Filter by jobTitle of veterinary establishment user
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortEntryNumber sort by entryNumber
     * @var string $sortName sort by name
     * @var string $sortPrivilegedCode sort by privilegedCode
     * @var string $sortType sort by type
     * @var string $sortUpdatedDate sort by updatedDate
     * @var string $sortVeterinaryEstablishmentAddressPostalCode Sort by postalCode of veterinaryEstablishmentAddress.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\SearchVeterinaryEstablishmentBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function searchVeterinaryEstablishment(Model\SearchRequest $requestBody, array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\SearchVeterinaryEstablishment($requestBody, $queryParameters, $headerParameters), $fetch);
    }

    /**
     * List Veterinary Establishment Addresses.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortPostalCode sort by postalCode
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentAddressResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllVeterinaryEstablishmentAddressBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllVeterinaryEstablishmentAddress(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllVeterinaryEstablishmentAddress($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a Veterinary Establishment Address.
     * This endpoint will create a Veterinary Establishment Address and add it in the Veterinary Establishment Address list.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentAddressResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostVeterinaryEstablishmentAddressBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postVeterinaryEstablishmentAddress(Model\VeterinaryEstablishmentAddressRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostVeterinaryEstablishmentAddress($requestBody, $headerParameters), $fetch);
    }

    /**
     * Delete a Veterinary Establishment Address.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\DeleteOneVeterinaryEstablishmentAddressBadRequestException
     * @throws Exception\DeleteOneVeterinaryEstablishmentAddressNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function deleteOneVeterinaryEstablishmentAddress(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\DeleteOneVeterinaryEstablishmentAddress($id, $headerParameters), $fetch);
    }

    /**
     * Get a Veterinary Establishment Address.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentAddressResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetOneVeterinaryEstablishmentAddressBadRequestException
     * @throws Exception\GetOneVeterinaryEstablishmentAddressNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getOneVeterinaryEstablishmentAddress(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetOneVeterinaryEstablishmentAddress($id, $headerParameters), $fetch);
    }

    /**
     * Update a Veterinary Establishment Address.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentAddressResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneVeterinaryEstablishmentAddressBadRequestException
     * @throws Exception\UpdateOneVeterinaryEstablishmentAddressNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneVeterinaryEstablishmentAddress(string $id, Model\SchemasVeterinaryEstablishmentAddressRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneVeterinaryEstablishmentAddress($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * List Veterinary Establishment Members.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $veterinaryEstablishmentId[] Filter by veterinaryEstablishmentId
     * @var array $veterinaryEstablishmentName[] Filter by veterinaryEstablishmentName
     * @var array $veterinaryEstablishmentPrivilegedCode[] Filter by veterinaryEstablishmentPrivilegedCode
     * @var array $veterinaryGroupId[] Filter by veterinaryGroupId
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate sort by updatedDate
     * @var string $sortVeterinaryEstablishmentName Sort by veterinaryEstablishmentName.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentMemberResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllVeterinaryEstablishmentMemberBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllVeterinaryEstablishmentMember(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllVeterinaryEstablishmentMember($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a Veterinary Establishment Member.
     * A Veterinary Establishment Member is the association of an Veterinary Establishment within a Veterinary Group.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentMemberResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostVeterinaryEstablishmentMemberBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postVeterinaryEstablishmentMember(Model\VeterinaryEstablishmentMemberRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostVeterinaryEstablishmentMember($requestBody, $headerParameters), $fetch);
    }

    /**
     * Delete a Veterinary Establishment Member.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\DeleteOneVeterinaryEstablishmentMemberBadRequestException
     * @throws Exception\DeleteOneVeterinaryEstablishmentMemberNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function deleteOneVeterinaryEstablishmentMember(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\DeleteOneVeterinaryEstablishmentMember($id, $headerParameters), $fetch);
    }

    /**
     * Update a Veterinary Establishment Member.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentMemberResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneVeterinaryEstablishmentMemberBadRequestException
     * @throws Exception\UpdateOneVeterinaryEstablishmentMemberNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneVeterinaryEstablishmentMember(string $id, Model\SchemasVeterinaryEstablishmentMemberRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneVeterinaryEstablishmentMember($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * Search on Veterinary Establishment Members.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $veterinaryEstablishmentId[] Filter by veterinaryEstablishmentId
     * @var array $veterinaryEstablishmentName[] Filter by veterinaryEstablishmentName
     * @var array $veterinaryEstablishmentPrivilegedCode[] Filter by veterinaryEstablishmentPrivilegedCode
     * @var array $veterinaryGroupId[] Filter by veterinaryGroupId
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate sort by updatedDate
     * @var string $sortVeterinaryEstablishmentName Sort by veterinaryEstablishmentName.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentMemberResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\SearchVeterinaryEstablishmentMemberBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function searchVeterinaryEstablishmentMember(Model\VeterinaryEstablishmentMemberSearchRequest $requestBody, array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\SearchVeterinaryEstablishmentMember($requestBody, $queryParameters, $headerParameters), $fetch);
    }

    /**
     * List Veterinary Establishment Activity.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $veterinaryEstablishmentId[] Filter by veterinaryEstablishmentId
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentActivityResponseItem[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllVeterinaryEstablishmentActivityBadRequestException
     * @throws Exception\GetAllVeterinaryEstablishmentActivityInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllVeterinaryEstablishmentActivity(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllVeterinaryEstablishmentActivity($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a Veterinary Establishment Activity.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\SchemasVeterinaryEstablishmentActivityResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostVeterinaryEstablishmentActivityBadRequestException
     * @throws Exception\PostVeterinaryEstablishmentActivityNotFoundException
     * @throws Exception\PostVeterinaryEstablishmentActivityConflictException
     * @throws Exception\PostVeterinaryEstablishmentActivityInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postVeterinaryEstablishmentActivity(Model\VeterinaryEstablishmentActivityRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostVeterinaryEstablishmentActivity($requestBody, $headerParameters), $fetch);
    }

    /**
     * Update a Veterinary Establishment Activity.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ComponentsSchemasVeterinaryEstablishmentActivityResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneVeterinaryEstablishmentActivityBadRequestException
     * @throws Exception\UpdateOneVeterinaryEstablishmentActivityNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneVeterinaryEstablishmentActivity(string $id, Model\SchemasVeterinaryEstablishmentActivityRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneVeterinaryEstablishmentActivity($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * List Veterinary Establishment Mandates.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $mandateId[] Filter by mandateId
     * @var array $veterinaryEstablishmentId[] Filter by veterinaryEstablishmentId
     * @var array $accountingCodePrefix[] Filter by accountingCodePrefix
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortAccountingCode sort by AccountingCode
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentMandateResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllVeterinaryEstablishmentMandateBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllVeterinaryEstablishmentMandate(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllVeterinaryEstablishmentMandate($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a Veterinary Establishment Mandate.
     * This endpoint will also create a Veterinary Establishment Mandate.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentMandateResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostVeterinaryEstablishmentMandateBadRequestException
     * @throws Exception\PostVeterinaryEstablishmentMandateNotFoundException
     * @throws Exception\PostVeterinaryEstablishmentMandateConflictException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postVeterinaryEstablishmentMandate(Model\VeterinaryEstablishmentMandateRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostVeterinaryEstablishmentMandate($requestBody, $headerParameters), $fetch);
    }

    /**
     * Get a Veterinary Establishment Mandate.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentMandateResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetOneVeterinaryEstablishmentMandateBadRequestException
     * @throws Exception\GetOneVeterinaryEstablishmentMandateNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getOneVeterinaryEstablishmentMandate(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetOneVeterinaryEstablishmentMandate($id, $headerParameters), $fetch);
    }

    /**
     * Update a Veterinary Establishment Mandate.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentMandateResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneVeterinaryEstablishmentMandateBadRequestException
     * @throws Exception\UpdateOneVeterinaryEstablishmentMandateNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneVeterinaryEstablishmentMandate(string $id, Model\SchemasVeterinaryEstablishmentMandateRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneVeterinaryEstablishmentMandate($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * List VeterinaryEstablishmentServicePeriods.
     * A VeterinaryEstablishmentServicePeriod represents a period, a contract with a fixed duration for an
     * establishment to a set of service options.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $veterinaryEstablishmentId[] Filter by veterinaryEstablishmentId
     * @var array $veterinaryEstablishmentServiceSubscriptionId[] Filter by veterinaryEstablishmentServiceSubscriptionId
     * @var bool  $shouldGenerateInvoice Filter by shouldGenerateInvoice
     * @var array $veterinaryServiceOptionCode[]
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromStartDate[] Returns results whose `startDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromEndDate[] Returns results whose `endDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toStartDate[] Returns results whose `startDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toEndDate[] Returns results whose `endDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate sort by updatedDate
     * @var string $sortStartDate sort by startDate
     * @var string $sortEndDate sort by endDate
     * @var string $sortVeterinaryServiceOptionEntitled Sort by veterinaryServiceOptionEntitled.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentServicePeriodResponseItem[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllVeterinaryEstablishmentServicePeriodBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllVeterinaryEstablishmentServicePeriod(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllVeterinaryEstablishmentServicePeriod($queryParameters, $headerParameters), $fetch);
    }

    /**
     * List VeterinaryEstablishmentServiceSubscriptions.
     * A VeterinaryEstablishmentServiceSubscription represents a subscription, a contract with a fixed duration for an
     * establishment to a set of service options.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $veterinaryEstablishmentId[] Filter by veterinaryEstablishmentId
     * @var array $veterinaryServiceOptionId[] Filter by veterinaryServiceOptionId
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromStartDate[] Returns results whose `startDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromEndDate[] Returns results whose `endDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toStartDate[] Returns results whose `startDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toEndDate[] Returns results whose `endDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate sort by updatedDate
     * @var string $sortVeterinaryServiceOptionEntitled Sort by veterinaryServiceOptionEntitled.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentServiceSubscriptionResponseItem[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllVeterinaryEstablishmentServiceSubscriptionBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllVeterinaryEstablishmentServiceSubscription(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllVeterinaryEstablishmentServiceSubscription($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a VeterinaryEstablishmentServiceSubscription.
     * A VeterinaryEstablishmentServiceSubscription represents a subscription, a contract with a fixed duration for an
     * establishment to a set of service options.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\SchemasVeterinaryEstablishmentServiceSubscriptionResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostVeterinaryEstablishmentServiceSubscriptionBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postVeterinaryEstablishmentServiceSubscription(Model\VeterinaryEstablishmentServiceSubscriptionRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostVeterinaryEstablishmentServiceSubscription($requestBody, $headerParameters), $fetch);
    }

    /**
     * Export Veterinary Establishment Services.
     *
     * @param array $queryParameters {
     *
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromCompletionDate[] Returns results whose `completionDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromEndDate[] Returns results whose `endDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromStartDate[] Returns results whose `startDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toCompletionDate[] Returns results whose `CompletionDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toEndDate[] Returns results whose `endDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toStartDate[] Returns results whose `startDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     *
     * }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\ExportVeterinaryEstablishmentServiceBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function exportVeterinaryEstablishmentService(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\ExportVeterinaryEstablishmentService($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Reduce a Veterinary Establishment Service Subscription.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentServiceSubscriptionReductionResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\ReduceOneVeterinaryEstablishmentServiceSubscriptionBadRequestException
     * @throws Exception\ReduceOneVeterinaryEstablishmentServiceSubscriptionNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function reduceOneVeterinaryEstablishmentServiceSubscription(string $id, Model\VeterinaryEstablishmentServiceSubscriptionReductionRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\ReduceOneVeterinaryEstablishmentServiceSubscription($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * Delete a Veterinary Establishment Service Subscription.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\DeleteOneVeterinaryEstablishmentServiceSubscriptionBadRequestException
     * @throws Exception\DeleteOneVeterinaryEstablishmentServiceSubscriptionNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function deleteOneVeterinaryEstablishmentServiceSubscription(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\DeleteOneVeterinaryEstablishmentServiceSubscription($id, $headerParameters), $fetch);
    }

    /**
     * Update a Veterinary Establishment Service Subscription.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ComponentsSchemasVeterinaryEstablishmentServiceSubscriptionResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneVeterinaryEstablishmentServiceSubscriptionBadRequestException
     * @throws Exception\UpdateOneVeterinaryEstablishmentServiceSubscriptionNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneVeterinaryEstablishmentServiceSubscription(string $id, Model\SchemasVeterinaryEstablishmentServiceSubscriptionRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneVeterinaryEstablishmentServiceSubscription($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * List VeterinaryEstablishmentServiceDiscounts.
     * A VeterinaryEstablishmentServiceDiscount represents a discount, which will reduce the subscription amount.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $veterinaryEstablishmentServicePeriodId[] Filter by veterinaryEstablishmentServicePeriodId
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentServiceDiscountResponseItem[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllVeterinaryEstablishmentServiceDiscountBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllVeterinaryEstablishmentServiceDiscount(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllVeterinaryEstablishmentServiceDiscount($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a Veterinary Establishment Service Discount.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\SchemasVeterinaryEstablishmentServiceDiscountResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postVeterinaryEstablishmentServiceDiscount(Model\VeterinaryEstablishmentServiceDiscountRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostVeterinaryEstablishmentServiceDiscount($requestBody, $headerParameters), $fetch);
    }

    /**
     * Delete a Veterinary Establishment Service Discount.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\DeleteOneVeterinaryEstablishmentServiceDiscountBadRequestException
     * @throws Exception\DeleteOneVeterinaryEstablishmentServiceDiscountNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function deleteOneVeterinaryEstablishmentServiceDiscount(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\DeleteOneVeterinaryEstablishmentServiceDiscount($id, $headerParameters), $fetch);
    }

    /**
     * Update a Veterinary Establishment Service Discount.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ComponentsSchemasVeterinaryEstablishmentServiceDiscountResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneVeterinaryEstablishmentServiceDiscountBadRequestException
     * @throws Exception\UpdateOneVeterinaryEstablishmentServiceDiscountNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneVeterinaryEstablishmentServiceDiscount(string $id, Model\SchemasVeterinaryEstablishmentServiceDiscountRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneVeterinaryEstablishmentServiceDiscount($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * List VeterinaryEstablishmentServiceDeliveryDiscounts.
     * A VeterinaryEstablishmentServiceDeliveryDiscount represents a discount,
     * which will reduce the service delivery price.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $veterinaryEstablishmentServiceDeliveryId[] Filter by veterinaryEstablishmentServiceDeliveryId
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentServiceDeliveryDiscountResponseItem[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllVeterinaryEstablishmentServiceDeliveryDiscountBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllVeterinaryEstablishmentServiceDeliveryDiscount(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllVeterinaryEstablishmentServiceDeliveryDiscount($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a Veterinary Establishment Service Delivery Discount.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\SchemasVeterinaryEstablishmentServiceDeliveryDiscountResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postVeterinaryEstablishmentServiceDeliveryDiscount(Model\VeterinaryEstablishmentServiceDeliveryDiscountRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostVeterinaryEstablishmentServiceDeliveryDiscount($requestBody, $headerParameters), $fetch);
    }

    /**
     * Delete a Veterinary Establishment Service Delivery Discount.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\DeleteOneVeterinaryEstablishmentServiceDeliveryDiscountBadRequestException
     * @throws Exception\DeleteOneVeterinaryEstablishmentServiceDeliveryDiscountNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function deleteOneVeterinaryEstablishmentServiceDeliveryDiscount(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\DeleteOneVeterinaryEstablishmentServiceDeliveryDiscount($id, $headerParameters), $fetch);
    }

    /**
     * Update a Veterinary Establishment Service Delivery Discount.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ComponentsSchemasVeterinaryEstablishmentServiceDeliveryDiscountResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneVeterinaryEstablishmentServiceDeliveryDiscountBadRequestException
     * @throws Exception\UpdateOneVeterinaryEstablishmentServiceDeliveryDiscountNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneVeterinaryEstablishmentServiceDeliveryDiscount(string $id, Model\SchemasVeterinaryEstablishmentServiceDeliveryDiscountRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneVeterinaryEstablishmentServiceDeliveryDiscount($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * List Veterinary Establishment Labellings.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $veterinaryEstablishmentId[] Filter by veterinaryEstablishmentId
     * @var array $veterinaryEstablishmentTagId[] Filter by veterinaryEstablishmentTagId
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentLabellingResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllVeterinaryEstablishmentLabellingBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllVeterinaryEstablishmentLabelling(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllVeterinaryEstablishmentLabelling($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a Veterinary Establishment Labelling.
     * A Veterinary Establishment Labelling is the association of an Veterinary Establishment within a Veterinary Group.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentLabellingResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostVeterinaryEstablishmentLabellingBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postVeterinaryEstablishmentLabelling(Model\VeterinaryEstablishmentLabellingRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostVeterinaryEstablishmentLabelling($requestBody, $headerParameters), $fetch);
    }

    /**
     * Delete a Veterinary Establishment Labelling.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\DeleteOneVeterinaryEstablishmentLabellingBadRequestException
     * @throws Exception\DeleteOneVeterinaryEstablishmentLabellingNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function deleteOneVeterinaryEstablishmentLabelling(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\DeleteOneVeterinaryEstablishmentLabelling($id, $headerParameters), $fetch);
    }

    /**
     * List Veterinary Establishment Tags.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $entitled[] Filter by entitled
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentTagResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllVeterinaryEstablishmentTagBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllVeterinaryEstablishmentTag(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllVeterinaryEstablishmentTag($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a Veterinary Establishment Tag.
     * This endpoint will also create a Veterinary Establishment Tag.
     * A tag makes it possible to categorise a set of veterinary establishments around a keyword.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentTagResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostVeterinaryEstablishmentTagBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postVeterinaryEstablishmentTag(Model\VeterinaryEstablishmentTagRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostVeterinaryEstablishmentTag($requestBody, $headerParameters), $fetch);
    }

    /**
     * Get a Veterinary Establishment Tag.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentTagResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetOneVeterinaryEstablishmentTagBadRequestException
     * @throws Exception\GetOneVeterinaryEstablishmentTagNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getOneVeterinaryEstablishmentTag(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetOneVeterinaryEstablishmentTag($id, $headerParameters), $fetch);
    }

    /**
     * List Veterinary Groups.
     * A Veterinary Group is a network of Veterinary Establishments.
     * A Veterinary Establishment may belong to several Veterinary Groups.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $legalStructureId[] Filter by legalStructureId
     * @var array $veterinaryEstablishmentId[] Filter by veterinaryEstablishmentId
     * @var array $veterinaryEstablishmentMemberId[] Filter by veterinaryEstablishmentMemberId
     * @var array $vguUserId[] Filter by userId of Veterinary Group User
     * @var array $entitled[] Filter by entitled
     * @var array $groupCode[] Filter by groupCode
     * @var array $status[]
     * @var array $vguUser[] Filter by jobTitle of veterinary group user
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortEntryNumber sort by entryNumber
     * @var string $sortGroupCode sort by groupCode
     * @var string $sortName sort by name
     * @var string $sortType sort by type
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryGroupResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllVeterinaryGroupBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllVeterinaryGroup(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllVeterinaryGroup($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a Veterinary Group.
     * A Veterinary Group is a network of Veterinary Establishments.
     * A Veterinary Establishment may belong to several Veterinary Groups.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryGroupResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostVeterinaryGroupBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postVeterinaryGroup(Model\VeterinaryGroupRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostVeterinaryGroup($requestBody, $headerParameters), $fetch);
    }

    /**
     * Get a Veterinary Group.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryGroupResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetOneVeterinaryGroupBadRequestException
     * @throws Exception\GetOneVeterinaryGroupNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getOneVeterinaryGroup(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetOneVeterinaryGroup($id, $headerParameters), $fetch);
    }

    /**
     * Update a Veterinary Group.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryGroupResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneVeterinaryGroupNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneVeterinaryGroup(string $id, Model\SchemasVeterinaryGroupRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneVeterinaryGroup($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * Search on Veterinary Groups.
     * A veterinary group is a company, enterprise, association identifiable by a legalCode.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $legalStructureId[] Filter by legalStructureId
     * @var array $veterinaryEstablishmentId[] Filter by veterinaryEstablishmentId
     * @var array $veterinaryEstablishmentMemberId[] Filter by veterinaryEstablishmentMemberId
     * @var array $vguUserId[] Filter by userId of Veterinary Group User
     * @var array $entitled[] Filter by entitled
     * @var array $groupCode[] Filter by groupCode
     * @var array $status[]
     * @var array $vguUser[] Filter by jobTitle of veterinary group user
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortEntryNumber sort by entryNumber
     * @var string $sortGroupCode sort by groupCode
     * @var string $sortName sort by name
     * @var string $sortType sort by type
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryGroupResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\SearchVeterinaryGroupBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function searchVeterinaryGroup(Model\VeterinaryGroupSearchRequest $requestBody, array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\SearchVeterinaryGroup($requestBody, $queryParameters, $headerParameters), $fetch);
    }

    /**
     * List Veterinary Group Mandates.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $mandateId[] Filter by mandateId
     * @var array $veterinaryGroupId[] Filter by veterinaryGroupId
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryGroupMandateResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllVeterinaryGroupMandateBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllVeterinaryGroupMandate(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllVeterinaryGroupMandate($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a Veterinary Group Mandate.
     * This endpoint will also create a Veterinary Group Mandate.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryGroupMandateResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostVeterinaryGroupMandateBadRequestException
     * @throws Exception\PostVeterinaryGroupMandateNotFoundException
     * @throws Exception\PostVeterinaryGroupMandateConflictException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postVeterinaryGroupMandate(Model\VeterinaryGroupMandateRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostVeterinaryGroupMandate($requestBody, $headerParameters), $fetch);
    }

    /**
     * Get a Veterinary Group Mandate.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryGroupMandateResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetOneVeterinaryGroupMandateBadRequestException
     * @throws Exception\GetOneVeterinaryGroupMandateNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getOneVeterinaryGroupMandate(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetOneVeterinaryGroupMandate($id, $headerParameters), $fetch);
    }

    /**
     * Update a Veterinary Group Mandate.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryGroupMandateResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneVeterinaryGroupMandateBadRequestException
     * @throws Exception\UpdateOneVeterinaryGroupMandateNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneVeterinaryGroupMandate(string $id, Model\SchemasVeterinaryGroupMandateRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneVeterinaryGroupMandate($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * List Veterinary Services.
     * A veterinary service could be MyVetshop, Crockit, VetoPartner...
     * Each veterinary service proposes one or more options that can be subscribed to by an veterinary establishment or
     * a veterinary group. A service cannot be created through the api as it requires integration of the service into the
     * reef eco-system.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $code[] Filter by code
     * @var array $veterinaryServiceOptionStatus[]
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCode sort by code
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryServiceResponseItem[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllVeterinaryServiceBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllVeterinaryService(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllVeterinaryService($queryParameters, $headerParameters), $fetch);
    }

    /**
     * List Veterinary Service Options.
     * A veterinary service option could be access to an online shop, to negotiating pricing.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $veterinaryServiceId[] Filter by veterinaryServiceId
     * @var array $code[] Filter by code
     * @var array $status[]
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCode sort by code
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate sort by updatedDate
     * @var string $sortEntitled Sort by Entitled.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryServiceOptionResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllVeterinaryServiceOptionBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllVeterinaryServiceOption(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllVeterinaryServiceOption($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Update a Veterinary Service Option.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryServiceOptionResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneVeterinaryServiceOptionBadRequestException
     * @throws Exception\UpdateOneVeterinaryServiceOptionNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneVeterinaryServiceOption(string $id, Model\VeterinaryServiceOptionRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneVeterinaryServiceOption($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * List VeterinaryServiceOptionConfigs.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $veterinaryServiceOptionId[] Filter by veterinaryServiceOptionId
     * @var array $code[] Filter by code
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $context Define the context. By default, the CREATE context is used.
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate sort by updatedDate
     * @var string $sortCode Sort by code.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryServiceOptionConfigResponseItem[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllVeterinaryServiceOptionConfigBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllVeterinaryServiceOptionConfig(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllVeterinaryServiceOptionConfig($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a You Sign Contract Confirmation.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\PostYouSignContractConfirmationBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postYouSignContractConfirmation(Model\YouSignContractConfirmationRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostYouSignContractConfirmation($requestBody, $headerParameters), $fetch);
    }

    /**
     * List Legacy Veterinary Establishment Credit Notes.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $moderationUserId[] Filter by moderationUserId
     * @var array $veterinaryEstablishmentId[] Filter by veterinaryEstablishmentId
     * @var array $documentCode[] Filter by documentCode
     * @var array $accountingStatus[] Filter by accountingStatus
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\VeterinaryEstablishmentCreditNoteResponseItem[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllLegacyVeterinaryEstablishmentCreditNoteBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllLegacyVeterinaryEstablishmentCreditNote(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllLegacyVeterinaryEstablishmentCreditNote($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a Legacy Veterinary Establishment Credit Note.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\SchemasVeterinaryEstablishmentCreditNoteResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostLegacyVeterinaryEstablishmentCreditNoteBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postLegacyVeterinaryEstablishmentCreditNote(Model\SchemasVeterinaryEstablishmentCreditNoteRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostLegacyVeterinaryEstablishmentCreditNote($requestBody, $headerParameters), $fetch);
    }

    /**
     * Update a Veterinary Establishment Credit Note.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ComponentsSchemasVeterinaryEstablishmentCreditNoteResponse|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneVeterinaryEstablishmentCreditNoteBadRequestException
     * @throws Exception\UpdateOneVeterinaryEstablishmentCreditNoteNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneVeterinaryEstablishmentCreditNote(string $id, Model\ComponentsSchemasVeterinaryEstablishmentCreditNoteRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneVeterinaryEstablishmentCreditNote($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * Update a lot of LegacyVeterinary Establishment Credit Note.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\UpdateLegacyVeterinaryEstablishmentCreditNoteBulkBadRequestException
     * @throws Exception\UpdateLegacyVeterinaryEstablishmentCreditNoteBulkNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateLegacyVeterinaryEstablishmentCreditNoteBulk(Model\VeterinaryEstablishmentCreditNoteBulkRequest $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateLegacyVeterinaryEstablishmentCreditNoteBulk($requestBody, $headerParameters), $fetch);
    }

    /**
     * Export Legacy Veterinary Establishment Credit Notes.
     *
     * @param array $queryParameters {
     *
     * @var array $id[] Filter by id
     * @var array $veterinaryEstablishmentId[] Filter by veterinaryEstablishmentId
     * @var array $moderationUserId[] Filter by moderationUserId
     * @var array $accountingStatus[] Filter by accountingStatus
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\ExportLegacyVeterinaryEstablishmentCreditNoteBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function exportLegacyVeterinaryEstablishmentCreditNote(array $queryParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\ExportLegacyVeterinaryEstablishmentCreditNote($queryParameters), $fetch);
    }

    /**
     * List Contact.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $customerId[] Filter by customerId
     * @var array $customerProfileId[] Filter by customerProfileId
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1ContactsGetResponse200Item[]|Model\V1ContactsGetResponse206Item[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllContactBadRequestException
     * @throws Exception\GetAllContactInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllContact(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllContact($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a Contact.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1ContactsPostResponse201|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostContactBadRequestException
     * @throws Exception\PostContactNotFoundException
     * @throws Exception\PostContactConflictException
     * @throws Exception\PostContactInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postContact(Model\V1ContactsPostBody $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostContact($requestBody, $headerParameters), $fetch);
    }

    /**
     * Delete a Contact.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\DeleteOneContactBadRequestException
     * @throws Exception\DeleteOneContactNotFoundException
     * @throws Exception\DeleteOneContactInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function deleteOneContact(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\DeleteOneContact($id, $headerParameters), $fetch);
    }

    /**
     * Update a Contact by ID.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1ContactsIdPatchResponse200|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneContactBadRequestException
     * @throws Exception\UpdateOneContactNotFoundException
     * @throws Exception\UpdateOneContactConflictException
     * @throws Exception\UpdateOneContactInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneContact(string $id, Model\V1ContactsIdPatchBody $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneContact($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * List Customer.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1CustomersGetResponse200Item[]|Model\V1CustomersGetResponse206Item[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllCustomerBadRequestException
     * @throws Exception\GetAllCustomerInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllCustomer(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllCustomer($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Update a Customer by ID.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1CustomersIdPatchResponse200|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneCustomerBadRequestException
     * @throws Exception\UpdateOneCustomerNotFoundException
     * @throws Exception\UpdateOneCustomerInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneCustomer(string $id, Model\V1CustomersIdPatchBody $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneCustomer($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * List Customer Profiles.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $customerId[] Filter by customerId
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1CustomerProfilesGetResponse200Item[]|Model\V1CustomerProfilesGetResponse206Item[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllCustomerProfileBadRequestException
     * @throws Exception\GetAllCustomerProfileInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllCustomerProfile(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllCustomerProfile($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create a Customer Profile.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1CustomerProfilesPostResponse201|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostCustomerProfileBadRequestException
     * @throws Exception\PostCustomerProfileNotFoundException
     * @throws Exception\PostCustomerProfileConflictException
     * @throws Exception\PostCustomerProfileInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postCustomerProfile(Model\V1CustomerProfilesPostBody $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostCustomerProfile($requestBody, $headerParameters), $fetch);
    }

    /**
     * Delete a Customer Profile.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\DeleteOneCustomerProfileBadRequestException
     * @throws Exception\DeleteOneCustomerProfileNotFoundException
     * @throws Exception\DeleteOneCustomerProfileInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function deleteOneCustomerProfile(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\DeleteOneCustomerProfile($id, $headerParameters), $fetch);
    }

    /**
     * Update a Customer Profile by ID.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1CustomerProfilesIdPatchResponse200|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOneCustomerProfileBadRequestException
     * @throws Exception\UpdateOneCustomerProfileNotFoundException
     * @throws Exception\UpdateOneCustomerProfileConflictException
     * @throws Exception\UpdateOneCustomerProfileInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOneCustomerProfile(string $id, Model\V1CustomerProfilesIdPatchBody $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOneCustomerProfile($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * List Pets.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetsGetResponse200Item[]|Model\V1PetsGetResponse206Item[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllPetBadRequestException
     * @throws Exception\GetAllPetInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllPet(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllPet($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create an Pet.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetsPostResponse201|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostPetBadRequestException
     * @throws Exception\PostPetInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postPet(Model\V1PetsPostBody $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostPet($requestBody, $headerParameters), $fetch);
    }

    /**
     * Update a Pet by ID.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetsIdPatchResponse200|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOnePetBadRequestException
     * @throws Exception\UpdateOnePetNotFoundException
     * @throws Exception\UpdateOnePetInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOnePet(string $id, Model\V1PetsIdPatchBody $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOnePet($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * List Pet Breeds.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $code[] Filter by code
     * @var array $name[] Filter by name
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetBreedsGetResponse200Item[]|Model\V1PetBreedsGetResponse206Item[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllPetBreedBadRequestException
     * @throws Exception\GetAllPetBreedInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllPetBreed(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllPetBreed($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create an PetBreed.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetBreedsPostResponse201|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostPetBreedBadRequestException
     * @throws Exception\PostPetBreedInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postPetBreed(Model\V1PetBreedsPostBody $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostPetBreed($requestBody, $headerParameters), $fetch);
    }

    /**
     * Delete a Pet Breed.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\DeleteOnePetBreedBadRequestException
     * @throws Exception\DeleteOnePetBreedNotFoundException
     * @throws Exception\DeleteOnePetBreedInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function deleteOnePetBreed(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\DeleteOnePetBreed($id, $headerParameters), $fetch);
    }

    /**
     * List Pet Breed External Links.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $petBreedId[] Filter by petBreedId
     * @var array $externalRefId[] Filter by externalRefId
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetBreedExternalLinksGetResponse200Item[]|Model\V1PetBreedExternalLinksGetResponse206Item[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllPetBreedExternalLinkBadRequestException
     * @throws Exception\GetAllPetBreedExternalLinkInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllPetBreedExternalLink(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllPetBreedExternalLink($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create an PetBreedExternalLink.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetBreedExternalLinksPostResponse201|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostPetBreedExternalLinkBadRequestException
     * @throws Exception\PostPetBreedExternalLinkInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postPetBreedExternalLink(Model\V1PetBreedExternalLinksPostBody $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostPetBreedExternalLink($requestBody, $headerParameters), $fetch);
    }

    /**
     * Delete a Pet Breed External Link.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\DeleteOnePetBreedExternalLinkBadRequestException
     * @throws Exception\DeleteOnePetBreedExternalLinkNotFoundException
     * @throws Exception\DeleteOnePetBreedExternalLinkInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function deleteOnePetBreedExternalLink(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\DeleteOnePetBreedExternalLink($id, $headerParameters), $fetch);
    }

    /**
     * List Pet Breed External Refs.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $name[] Filter by name
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetBreedExternalRefsGetResponse200Item[]|Model\V1PetBreedExternalRefsGetResponse206Item[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllPetBreedExternalRefBadRequestException
     * @throws Exception\GetAllPetBreedExternalRefInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllPetBreedExternalRef(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllPetBreedExternalRef($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create an PetBreedExternalRef.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetBreedExternalRefsPostResponse201|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostPetBreedExternalRefBadRequestException
     * @throws Exception\PostPetBreedExternalRefInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postPetBreedExternalRef(Model\V1PetBreedExternalRefsPostBody $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostPetBreedExternalRef($requestBody, $headerParameters), $fetch);
    }

    /**
     * Delete a Pet Breed External Ref.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\DeleteOnePetBreedExternalRefBadRequestException
     * @throws Exception\DeleteOnePetBreedExternalRefNotFoundException
     * @throws Exception\DeleteOnePetBreedExternalRefInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function deleteOnePetBreedExternalRef(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\DeleteOnePetBreedExternalRef($id, $headerParameters), $fetch);
    }

    /**
     * List Pet Breed Labellings.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $petBreedId[] Filter by petBreedId
     * @var array $tagId[] Filter by tagId
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetBreedLabellingsGetResponse200Item[]|Model\V1PetBreedLabellingsGetResponse206Item[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllPetBreedLabellingBadRequestException
     * @throws Exception\GetAllPetBreedLabellingInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllPetBreedLabelling(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllPetBreedLabelling($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create an PetBreedLabelling.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetBreedLabellingsPostResponse201|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostPetBreedLabellingBadRequestException
     * @throws Exception\PostPetBreedLabellingInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postPetBreedLabelling(Model\V1PetBreedLabellingsPostBody $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostPetBreedLabelling($requestBody, $headerParameters), $fetch);
    }

    /**
     * Delete a Pet Breed Labelling.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\DeleteOnePetBreedLabellingBadRequestException
     * @throws Exception\DeleteOnePetBreedLabellingNotFoundException
     * @throws Exception\DeleteOnePetBreedLabellingInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function deleteOnePetBreedLabelling(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\DeleteOnePetBreedLabelling($id, $headerParameters), $fetch);
    }

    /**
     * List Pet Breed Tags.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $name[] Filter by name
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetBreedTagsGetResponse200Item[]|Model\V1PetBreedTagsGetResponse206Item[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllPetBreedTagBadRequestException
     * @throws Exception\GetAllPetBreedTagInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllPetBreedTag(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllPetBreedTag($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create an PetBreedTag.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetBreedTagsPostResponse201|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostPetBreedTagBadRequestException
     * @throws Exception\PostPetBreedTagInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postPetBreedTag(Model\V1PetBreedTagsPostBody $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostPetBreedTag($requestBody, $headerParameters), $fetch);
    }

    /**
     * Delete a Pet Breed Tag.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\DeleteOnePetBreedTagBadRequestException
     * @throws Exception\DeleteOnePetBreedTagNotFoundException
     * @throws Exception\DeleteOnePetBreedTagInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function deleteOnePetBreedTag(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\DeleteOnePetBreedTag($id, $headerParameters), $fetch);
    }

    /**
     * Create an Pet Microchip.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetMicrochipsPostResponse201|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostPetMicrochipBadRequestException
     * @throws Exception\PostPetMicrochipInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postPetMicrochip(Model\V1PetMicrochipsPostBody $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostPetMicrochip($requestBody, $headerParameters), $fetch);
    }

    /**
     * Delete a Pet Microchip.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\DeleteOnePetMicrochipBadRequestException
     * @throws Exception\DeleteOnePetMicrochipNotFoundException
     * @throws Exception\DeleteOnePetMicrochipInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function deleteOnePetMicrochip(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\DeleteOnePetMicrochip($id, $headerParameters), $fetch);
    }

    /**
     * List Pet Profiles.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $type[] Filter by pet profile type
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetProfilesGetResponse200Item[]|Model\V1PetProfilesGetResponse206Item[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllPetProfileBadRequestException
     * @throws Exception\GetAllPetProfileInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllPetProfile(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllPetProfile($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Create an PetCustomerProfile.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetCustomerProfilesPostResponse201|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostPetCustomerProfileBadRequestException
     * @throws Exception\PostPetCustomerProfileInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postPetCustomerProfile(Model\V1PetCustomerProfilesPostBody $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostPetCustomerProfile($requestBody, $headerParameters), $fetch);
    }

    /**
     * Update a Pet Customer Profile by ID.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetCustomerProfilesIdPatchResponse200|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOnePetCustomerProfileBadRequestException
     * @throws Exception\UpdateOnePetCustomerProfileNotFoundException
     * @throws Exception\UpdateOnePetCustomerProfileInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOnePetCustomerProfile(string $id, Model\V1PetCustomerProfilesIdPatchBody $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOnePetCustomerProfile($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * Create an PetGhostProfile.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetGhostProfilesPostResponse201|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostPetGhostProfileBadRequestException
     * @throws Exception\PostPetGhostProfileInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postPetGhostProfile(Model\V1PetGhostProfilesPostBody $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostPetGhostProfile($requestBody, $headerParameters), $fetch);
    }

    /**
     * Update a Pet Ghost Profile by ID.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetGhostProfilesIdPatchResponse200|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOnePetGhostProfileBadRequestException
     * @throws Exception\UpdateOnePetGhostProfileNotFoundException
     * @throws Exception\UpdateOnePetGhostProfileInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOnePetGhostProfile(string $id, Model\V1PetGhostProfilesIdPatchBody $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOnePetGhostProfile($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * Create an PetPatientProfile.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetPatientProfilesPostResponse201|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostPetPatientProfileBadRequestException
     * @throws Exception\PostPetPatientProfileInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postPetPatientProfile(Model\V1PetPatientProfilesPostBody $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostPetPatientProfile($requestBody, $headerParameters), $fetch);
    }

    /**
     * Update a Pet Patient Profile by ID.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetPatientProfilesIdPatchResponse200|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOnePetPatientProfileBadRequestException
     * @throws Exception\UpdateOnePetPatientProfileNotFoundException
     * @throws Exception\UpdateOnePetPatientProfileInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOnePetPatientProfile(string $id, Model\V1PetPatientProfilesIdPatchBody $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOnePetPatientProfile($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * Create an PetPublicProfile.
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetPublicProfilesPostResponse201|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostPetPublicProfileBadRequestException
     * @throws Exception\PostPetPublicProfileInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postPetPublicProfile(Model\V1PetPublicProfilesPostBody $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostPetPublicProfile($requestBody, $headerParameters), $fetch);
    }

    /**
     * Update a Pet Public Profile by ID.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetPublicProfilesIdPatchResponse200|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UpdateOnePetPublicProfileBadRequestException
     * @throws Exception\UpdateOnePetPublicProfileNotFoundException
     * @throws Exception\UpdateOnePetPublicProfileInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function updateOnePetPublicProfile(string $id, Model\V1PetPublicProfilesIdPatchBody $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\UpdateOnePetPublicProfile($id, $requestBody, $headerParameters), $fetch);
    }

    /**
     * List Pet Species.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $id[] Filter by id
     * @var array $name[] Filter by name
     * @var array $category[] Filter by category
     * @var array $code[] Filter by code
     * @var array $fromCreatedDate[] Returns results whose `createdDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $fromUpdatedDate[] Returns results whose `updatedDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toCreatedDate[] Returns results whose `createdDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var array $toUpdatedDate[] Returns results whose `updatedDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortCreatedDate sort by createdDate
     * @var string $sortUpdatedDate Sort by updatedDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\V1PetSpeciesGetResponse200Item[]|Model\V1PetSpeciesGetResponse206Item[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllPetSpecieBadRequestException
     * @throws Exception\GetAllPetSpecieInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllPetSpecie(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllPetSpecie($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Retrieves a Client resource.
     *
     * @param string $slug   Resource identifier
     * @param string $fetch  Fetch mode to use (can be OBJECT or RESPONSE)
     * @param array  $accept Accept content header application/json|text/html
     *
     * @return Model\ClientGet|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetClientItemNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getClientItem(string $slug, string $fetch = self::FETCH_OBJECT, array $accept = [])
    {
        return $this->executeEndpoint(new Endpoint\GetClientItem($slug, $accept), $fetch);
    }

    /**
     * Update client resources types and allowed permissions.
     *
     * @param string $slug             Resource identifier
     * @param array  $headerParameters {
     *
     * @var string $X-Reef-Client-ID Public OAuth Client ID
     * @var string $X-Reef-Client-HMAC HMAC computed using hash_hmac('sha256', bearer, clientSecret)
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\PutDefinitionClientItemBadRequestException
     * @throws Exception\PutDefinitionClientItemUnprocessableEntityException
     * @throws Exception\PutDefinitionClientItemNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function putDefinitionClientItem(string $slug, Model\ServiceDefinition $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PutDefinitionClientItem($slug, $requestBody, $headerParameters), $fetch);
    }

    /**
     * List policy results stored in the IAM.
     *
     * @param array $queryParameters {
     *
     * @var int    $page The collection page number
     * @var int    $itemsPerPage The number of items per page
     * @var string $name
     * @var array  $name[]
     * @var string $subject
     * @var array  $subject[]
     * @var string $etag
     * @var array  $etag[]
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $X-Reef-Client-ID Public OAuth Client ID
     * @var string $X-Reef-Client-HMAC HMAC computed using hash_hmac('sha256', bearer, clientSecret)
     *             }
     *
     * @param string $fetch  Fetch mode to use (can be OBJECT or RESPONSE)
     * @param array  $accept Accept content header application/json|text/html
     *
     * @return Model\PolicyResultPolicyResultGet[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getPolicyResultCollection(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT, array $accept = [])
    {
        return $this->executeEndpoint(new Endpoint\GetPolicyResultCollection($queryParameters, $headerParameters, $accept), $fetch);
    }

    /**
     * Store a policy result in the IAM.
     *
     * @param array $headerParameters {
     *
     * @var string $X-Reef-Client-ID Public OAuth Client ID
     * @var string $X-Reef-Client-HMAC HMAC computed using hash_hmac('sha256', bearer, clientSecret)
     *             }
     *
     * @param string $fetch  Fetch mode to use (can be OBJECT or RESPONSE)
     * @param array  $accept Accept content header application/json|text/html
     *
     * @return Model\PolicyResultPolicyResultGet|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostPolicyResultCollectionBadRequestException
     * @throws Exception\PostPolicyResultCollectionUnprocessableEntityException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postPolicyResultCollection(Model\PolicyResultPolicyResultPost $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT, array $accept = [])
    {
        return $this->executeEndpoint(new Endpoint\PostPolicyResultCollection($requestBody, $headerParameters, $accept), $fetch);
    }

    /**
     * Get a specific policy result.
     *
     * @param string $id               Resource identifier
     * @param array  $headerParameters {
     *
     * @var string $X-Reef-Client-ID Public OAuth Client ID
     * @var string $X-Reef-Client-HMAC HMAC computed using hash_hmac('sha256', bearer, clientSecret)
     *             }
     *
     * @param string $fetch  Fetch mode to use (can be OBJECT or RESPONSE)
     * @param array  $accept Accept content header application/json|text/html
     *
     * @return Model\PolicyResultPolicyResultGet|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetPolicyResultItemNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getPolicyResultItem(string $id, array $headerParameters = [], string $fetch = self::FETCH_OBJECT, array $accept = [])
    {
        return $this->executeEndpoint(new Endpoint\GetPolicyResultItem($id, $headerParameters, $accept), $fetch);
    }

    /**
     * Retrieves the collection of Principal resources.
     *
     * @param array $queryParameters {
     *
     * @var int    $page The collection page number
     * @var int    $itemsPerPage The number of items per page
     * @var string $email
     * @var string $type
     * @var array  $type[]
     * @var string $status
     * @var array  $status[]
     * @var string $id
     * @var string $id[]
     * @var string $order[id]
     * @var string $order[email]
     * @var string $order[created]
     * @var string $order[updated]
     * @var string $search
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\PrincipalGet[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getPrincipalCollection(array $queryParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetPrincipalCollection($queryParameters), $fetch);
    }

    /**
     * Invite a person to join the reef by validating a pre-created account.
     *
     * @param array $headerParameters {
     *
     * @var string $X-Reef-Client-ID Public OAuth Client ID
     * @var string $X-Reef-Client-HMAC HMAC computed using hash_hmac('sha256', bearer, clientSecret)
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\PrincipalGet|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostPrincipalCollectionBadRequestException
     * @throws Exception\PostPrincipalCollectionUnprocessableEntityException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postPrincipalCollection(Model\PrincipalShort $requestBody, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PostPrincipalCollection($requestBody, $headerParameters), $fetch);
    }

    /**
     * Retrieves a Principal resource.
     *
     * @param string $id    Resource identifier
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\PrincipalGet|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetPrincipalItemNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getPrincipalItem(string $id, string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetPrincipalItem($id), $fetch);
    }

    /**
     * Update a principal.
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\PrincipalGet|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PutPrincipalItemBadRequestException
     * @throws Exception\PutPrincipalItemUnprocessableEntityException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function putPrincipalItem(Model\PrincipalPut $requestBody, string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PutPrincipalItem($requestBody), $fetch);
    }

    /**
     * Create a resource group.
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ResourceGroupGet|\Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getResourceGroupCollection(Model\ResourceGroupPost $requestBody, string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetResourceGroupCollection($requestBody), $fetch);
    }

    /**
     * Read a resource group.
     *
     * @param string $slug  Resource group slug
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ResourceGroupGet|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetResourceGroupItemNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getResourceGroupItem(string $slug, string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetResourceGroupItem($slug), $fetch);
    }

    /**
     * Update a resource group.
     *
     * @param string $slug  Resource group slug
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ResourceGroupGet|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PutResourceGroupItemNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function putResourceGroupItem(string $slug, Model\ResourceGroupPut $requestBody, string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\PutResourceGroupItem($slug, $requestBody), $fetch);
    }

    /**
     * Retrieves a ResourceGroup resource.
     *
     * @param string $slug   Resource identifier
     * @param string $fetch  Fetch mode to use (can be OBJECT or RESPONSE)
     * @param array  $accept Accept content header application/json|text/html
     *
     * @return Model\ResourceGroupResourceGroupGet|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetPermissionsResourceGroupItemNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getPermissionsResourceGroupItem(string $slug, string $fetch = self::FETCH_OBJECT, array $accept = [])
    {
        return $this->executeEndpoint(new Endpoint\GetPermissionsResourceGroupItem($slug, $accept), $fetch);
    }

    /**
     * Retrieves a ResourceType resource.
     *
     * @param string $slug   Resource identifier
     * @param string $fetch  Fetch mode to use (can be OBJECT or RESPONSE)
     * @param array  $accept Accept content header application/json|text/html
     *
     * @return Model\ResourceTypeGetshort|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetResourceTypeItemNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getResourceTypeItem(string $slug, string $fetch = self::FETCH_OBJECT, array $accept = [])
    {
        return $this->executeEndpoint(new Endpoint\GetResourceTypeItem($slug, $accept), $fetch);
    }

    /**
     * Retrieve effective permissions on alle resources a the specified resource type.
     *
     * @param string $slug            Resource identifier
     * @param array  $queryParameters {
     *
     * @var string $principal Principal ID (if different from the one doing the query)
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $X-Reef-Client-ID Public OAuth Client ID
     * @var string $X-Reef-Client-HMAC HMAC computed using hash_hmac('sha256', bearer, clientSecret)
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\PermissionGrantList|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetDetailedPermissionsResourceTypeItemNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getDetailedPermissionsResourceTypeItem(string $slug, array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetDetailedPermissionsResourceTypeItem($slug, $queryParameters, $headerParameters), $fetch);
    }

    /**
     * Retrieves a ResourceType resource.
     *
     * @param string $slug            Resource identifier
     * @param array  $queryParameters {
     *
     * @var string $principal Principal ID (if different from the one doing the query)
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $X-Reef-Client-ID Public OAuth Client ID
     * @var string $X-Reef-Client-HMAC HMAC computed using hash_hmac('sha256', bearer, clientSecret)
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\PermissionGrantList|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetPermissionsResourceTypeItemNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getPermissionsResourceTypeItem(string $slug, array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetPermissionsResourceTypeItem($slug, $queryParameters, $headerParameters), $fetch);
    }

    /**
     * Retrieves the collection of Resource resources.
     *
     * @param array $queryParameters {
     *
     * @var int    $page The collection page number
     * @var int    $itemsPerPage The number of items per page
     * @var string $slug
     * @var array  $slug[]
     *             }
     *
     * @param string $fetch  Fetch mode to use (can be OBJECT or RESPONSE)
     * @param array  $accept Accept content header application/json|text/html
     *
     * @return Model\Resource[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getResourceCollection(array $queryParameters = [], string $fetch = self::FETCH_OBJECT, array $accept = [])
    {
        return $this->executeEndpoint(new Endpoint\GetResourceCollection($queryParameters, $accept), $fetch);
    }

    /**
     * Creates a Resource resource.
     *
     * @param string $fetch  Fetch mode to use (can be OBJECT or RESPONSE)
     * @param array  $accept Accept content header application/json|text/html
     *
     * @return Model\Resource|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PostResourceCollectionBadRequestException
     * @throws Exception\PostResourceCollectionUnprocessableEntityException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function postResourceCollection(Model\Resource $requestBody, string $fetch = self::FETCH_OBJECT, array $accept = [])
    {
        return $this->executeEndpoint(new Endpoint\PostResourceCollection($requestBody, $accept), $fetch);
    }

    /**
     * Retrieve effective permissions on a collection of resources.
     *
     * @param array $queryParameters {
     *
     * @var array  $slug[] List of resources you want permissions for
     * @var string $principal Principal ID (if different from the one doing the query)
     * @var int    $page The collection page number
     * @var int    $itemsPerPage The number of items per page
     * @var string $slug
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $X-Reef-Client-ID Public OAuth Client ID
     * @var string $X-Reef-Client-HMAC HMAC computed using hash_hmac('sha256', bearer, clientSecret)
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\PermissionGrantList|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getPermissionsResourceCollection(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetPermissionsResourceCollection($queryParameters, $headerParameters), $fetch);
    }

    /**
     * Retrieves a Resource resource.
     *
     * @param string $slug   Resource identifier
     * @param string $fetch  Fetch mode to use (can be OBJECT or RESPONSE)
     * @param array  $accept Accept content header application/json|text/html
     *
     * @return Model\Resource|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetResourceItemNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getResourceItem(string $slug, string $fetch = self::FETCH_OBJECT, array $accept = [])
    {
        return $this->executeEndpoint(new Endpoint\GetResourceItem($slug, $accept), $fetch);
    }

    /**
     * Replaces the Resource resource.
     *
     * @param string $slug   Resource identifier
     * @param string $fetch  Fetch mode to use (can be OBJECT or RESPONSE)
     * @param array  $accept Accept content header application/json|text/html
     *
     * @return Model\Resource|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\PutResourceItemBadRequestException
     * @throws Exception\PutResourceItemUnprocessableEntityException
     * @throws Exception\PutResourceItemNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function putResourceItem(string $slug, Model\Resource $requestBody, string $fetch = self::FETCH_OBJECT, array $accept = [])
    {
        return $this->executeEndpoint(new Endpoint\PutResourceItem($slug, $requestBody, $accept), $fetch);
    }

    /**
     * Retrieves a Resource resource.
     *
     * @param string $slug             Resource identifier
     * @param array  $headerParameters {
     *
     * @var string $X-Reef-Client-ID Public OAuth Client ID
     * @var string $X-Reef-Client-HMAC HMAC computed using hash_hmac('sha256', bearer, clientSecret)
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\PermissionGrant|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetDetailedPermissionsResourceItemNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getDetailedPermissionsResourceItem(string $slug, array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetDetailedPermissionsResourceItem($slug, $headerParameters), $fetch);
    }

    /**
     * Retrieves a Resource resource.
     *
     * @param string $slug            Resource identifier
     * @param array  $queryParameters {
     *
     * @var string $principal Principal ID (if different from the one doing the query)
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $X-Reef-Client-ID Public OAuth Client ID
     * @var string $X-Reef-Client-HMAC HMAC computed using hash_hmac('sha256', bearer, clientSecret)
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\PermissionGrant|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetPermissionsResourceItemNotFoundException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getPermissionsResourceItem(string $slug, array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetPermissionsResourceItem($slug, $queryParameters, $headerParameters), $fetch);
    }

    /**
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return \Psr\Http\Message\ResponseInterface|null
     *
     * @throws Exception\RevokeBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function revoke(string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\Revoke(), $fetch);
    }

    /**
     * Retrieve token info from token.
     *
     * @param string $token The token
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\OauthV2TokenInfoTokenGetResponse200|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\TokeninfoUnauthorizedException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function tokeninfo(string $token, string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\Tokeninfo($token), $fetch);
    }

    /**
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\UserinfoGetResponse200|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\UserinfoUnauthorizedException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function userinfo(string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\Userinfo(), $fetch);
    }

    /**
     * List supplier orders.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $veterinaryEstablishmentId[] Filter by veterinaryEstablishmentId
     * @var array $fromInvoiceDate Returns results whose `invoiceDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toInvoiceDate Returns results whose `invoiceDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortInvoiceDate Sort by invoiceDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\SupplierOrderResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllSupplierOrderBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllSupplierOrder(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllSupplierOrder($queryParameters, $headerParameters), $fetch);
    }

    /**
     * List web orders.
     *
     * @param array $queryParameters {
     *
     * @var int   $page the page of items to collect a part of result set
     * @var int   $limit the maximum number of items to collect a part of result set
     * @var array $veterinaryEstablishmentId[] Filter by veterinaryEstablishmentId
     * @var array $fromInvoiceDate Returns results whose `invoiceDate` is greater than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |----------Filtered---------|----------Returned----------|
     * @var array $toInvoiceDate Returns results whose `invoiceDate` is lower than or equal to the specified value.
     *            Example for 2022-03-15 value :
     *
     * |----Jan---|----Feb---|----Mar---|----Apr----|----May----|
     * |-----------Returned--------|---------Filtered-----------|
     * @var string $sortInvoiceDate Sort by invoiceDate.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\WebOrderResponse[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllWebOrderBadRequestException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllWebOrder(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllWebOrder($queryParameters, $headerParameters), $fetch);
    }

    /**
     * List Product Prices.
     *
     * @param array $queryParameters {
     *
     * @var int    $page the page of items to collect a part of result set
     * @var int    $limit the maximum number of items to collect a part of result set
     * @var array  $alcyonProductCode[] Filter by alcyonProductCode
     * @var array  $centravetProductCode[] Filter by centravetProductCode
     * @var array  $covetoProductCode[] Filter by covetoProductCode
     * @var array  $ean[] Filter by ean
     * @var array  $gtin[] Filter by gtin
     * @var array  $psProductAttributeId[] Filter by ps_product_attribute_id
     * @var array  $fromCreated[] Filter from created
     * @var array  $fromUpdated[] Filter from updated
     * @var array  $toCreated[] Filter to created
     * @var array  $toUpdated[] Filter to updated
     * @var string $sortCreated sort by created
     * @var string $sortUpdated Sort by updated.
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     *
     * @param string $fetch Fetch mode to use (can be OBJECT or RESPONSE)
     *
     * @return Model\ProductPriceResponseItem[]|\Psr\Http\Message\ResponseInterface
     *
     * @throws Exception\GetAllProductPriceBadRequestException
     * @throws Exception\GetAllProductPriceInternalServerErrorException
     * @throws Exception\UnexpectedStatusCodeException
     */
    public function getAllProductPrice(array $queryParameters = [], array $headerParameters = [], string $fetch = self::FETCH_OBJECT)
    {
        return $this->executeEndpoint(new Endpoint\GetAllProductPrice($queryParameters, $headerParameters), $fetch);
    }

    public static function create($httpClient = null, array $additionalPlugins = [], array $additionalNormalizers = [])
    {
        if (null === $httpClient) {
            $httpClient = \Http\Discovery\Psr18ClientDiscovery::find();
            $plugins = [];
            if (0 < \count($additionalPlugins)) {
                $plugins = \array_merge($plugins, $additionalPlugins);
            }
            $httpClient = new \Http\Client\Common\PluginClient($httpClient, $plugins);
        }
        $requestFactory = \Http\Discovery\Psr17FactoryDiscovery::findRequestFactory();
        $streamFactory = \Http\Discovery\Psr17FactoryDiscovery::findStreamFactory();
        $normalizers = [new \Symfony\Component\Serializer\Normalizer\ArrayDenormalizer(), new Normalizer\JaneObjectNormalizer()];
        if (0 < \count($additionalNormalizers)) {
            $normalizers = \array_merge($normalizers, $additionalNormalizers);
        }
        $serializer = new \Symfony\Component\Serializer\Serializer($normalizers, [new \Symfony\Component\Serializer\Encoder\JsonEncoder(new \Symfony\Component\Serializer\Encoder\JsonEncode(), new \Symfony\Component\Serializer\Encoder\JsonDecode(['json_decode_associative' => true]))]);

        return new static($httpClient, $requestFactory, $serializer, $streamFactory);
    }
}
