<?php

declare(strict_types=1);

namespace NoahVet\Reef\Iterator;

use NoahVet\Reef\Factory\Jane\EndpointFactoryInterface;
use NoahVet\Reef\Jane\Normalizer\JaneObjectNormalizer;
use NoahVet\Reef\Jane\Runtime\Client\Endpoint;
use Psr\Http\Message\ResponseInterface;
use Symfony\Component\Serializer\Encoder\JsonDecode;
use Symfony\Component\Serializer\Encoder\JsonEncode;
use Symfony\Component\Serializer\Encoder\JsonEncoder;
use Symfony\Component\Serializer\Normalizer\ArrayDenormalizer;
use Symfony\Component\Serializer\Serializer;

/**
 * @template TValue of object
 * @template TParams of array<string, mixed>
 * @template TEndpoint of Endpoint
 *
 * @extends AbstractSimpleReefApiIterator<TValue, TParams>
 */
class ApiClientSimpleReefIterator extends AbstractSimpleReefApiIterator
{
    private readonly Serializer $serializer;

    /**
     * @param class-string<TEndpoint> $endpointClass
     */
    public function __construct(
        private readonly object $client,
        private readonly string $endpointClass,
        private readonly EndpointFactoryInterface $endpointFactory,
        array $params,
        int $querySize = 50,
    ) {
        parent::__construct($params, $querySize);

        $this->serializer = new Serializer(
            [
                new ArrayDenormalizer(),
                new JaneObjectNormalizer(),
            ],
            [
                new JsonEncoder(
                    new JsonEncode(),
                    new JsonDecode(['json_decode_associative' => true],
                    ),
                ),
            ]);
    }

    public function apiCall(array $params): ResponseInterface
    {
        $endpoint = $this->endpointFactory->create($this->endpointClass, $params);

        \assert(\method_exists($this->client, 'executeRawEndpoint'));

        return $this->client->executeRawEndpoint($endpoint);
    }

    public function parseResponse(ResponseInterface $response): array
    {
        $endpoint = $this->endpointFactory->create($this->endpointClass, $this->params);

        return $endpoint->parseResponse($response, $this->serializer);
    }

    /**
     * @param TParams $params
     */
    protected function instanciateEndpoint(array $params): Endpoint
    {
        return new $this->endpointClass($params);
    }
}
