<?php

declare(strict_types=1);

namespace NoahVet\Reef\Domain\Tool;

class StringTool implements StringToolInterface
{
    public static function fromBurgerCaseToSnakeCase(string $subject): string
    {
        return \mb_strtolower(
            \str_replace(
                '-',
                '_',
                \trim($subject),
            ),
        );
    }

    public static function fromSnakeCaseToBurgerCase(string $subject): string
    {
        return \mb_strtolower(
            \str_replace(
                '_',
                '-',
                \trim($subject),
            ),
        );
    }

    public static function fromPascalCaseToCamelCase(string $subject): string
    {
        return \lcfirst(\trim($subject));
    }

    public static function fromSpacedToBurgerCase(string $subject): string
    {
        return \mb_strtolower(
            \str_replace(
                ' ',
                '-',
                \trim($subject),
            ),
        );
    }

    public static function fromSpacedToUpperBurgerCase(string $subject): string
    {
        return \mb_strtoupper(
            \str_replace(
                ' ',
                '-',
                \trim($subject),
            ),
        );
    }

    public static function fromSpacedToPascalCase(string $subject): string
    {
        return \ucwords(
            \mb_strtolower(
                \trim($subject),
            ),
        );
    }

    public static function toPascalCase(string $subject): string
    {
        $subject = \trim($subject);

        /** @var string[]|false $parts */
        $parts = \preg_split(
            '/(?<=\p{Ll})(?=\p{Lu})|[^[:alnum:]]+/u',
            $subject,
            -1,
            \PREG_SPLIT_NO_EMPTY,
        );

        if (false === $parts) {
            $parts = [];
        }

        $parts = \array_map(function (string $w): string {
            if (\preg_match('/^\p{Lu}{2,}\d*$/u', $w)) {
                return $w;
            }
            if (\preg_match('/^\p{Lu}\p{Lu}+/u', $w)) {
                return $w;
            }
            $w = \mb_strtolower($w, 'UTF-8');
            $first = \mb_substr($w, 0, 1, 'UTF-8');
            $rest = \mb_substr($w, 1, null, 'UTF-8');

            return \mb_strtoupper($first, 'UTF-8').$rest;
        }, $parts);

        return \implode('', $parts);
    }

    public static function fromSpacedToSnakeUpperCase(string $subject): string
    {
        return \mb_strtoupper(
            \str_replace(
                ' ',
                '_',
                \trim($subject),
            ),
        );
    }

    public static function toLowerCaseWithoutAccent(string $subject): string
    {
        return self::removeAccent(
            \mb_strtolower($subject),
        );
    }

    public static function toUpperCaseWithoutAccent(string $subject): string
    {
        return self::removeAccent(
            \mb_strtoupper($subject),
        );
    }

    public static function removeAccent(string $subject): string
    {
        return \str_replace(
            \mb_str_split(self::CHAR_LIST_WITH_ACCENT),
            \mb_str_split(self::LOWER_CASE_LIST_WITH_ACCENT.self::UPPER_CASE_LIST_WITH_ACCENT),
            \trim($subject),
        );
    }

    public static function removeWhitespace(string $subject): string
    {
        return \str_replace(
            ' ',
            '',
            $subject,
        );
    }

    public static function toName(string $name, array $supportedSeparators = self::SUPPORTED_SEPARATORS): string
    {
        $name = \ucwords(\mb_strtolower(\trim($name)));

        foreach ($supportedSeparators as $separator) {
            if (\str_contains($name, $separator) && !empty($separator)) {
                $namePart = \explode($separator, $name);
                $name = \implode(
                    $separator,
                    \array_map(
                        'ucfirst',
                        \array_map(
                            'mb_strtolower',
                            $namePart,
                        ),
                    ),
                );
            }
        }

        return $name;
    }

    /**
     * Convert camelCase/PascalCase to burger-case while preserving all-uppercase acronyms.
     *
     * Examples:
     *  - "camelCase"        -> "camel-case"
     *  - "PascalCase"       -> "pascal-case"
     *  - "HTTPResponse2XX"  -> "HTTP-response-2-xx"
     *  - " aABáâCãDä "      -> "a-A-báâ-cã-dä"
     */
    public static function fromCamelCaseToBurgerCase(string $subject): string
    {
        // Split acronyms followed by Cap+lower sequences: "HTTPResponse" -> "HTTP-Response"
        $s = self::safePregReplace('/(\p{Lu}+)(\p{Lu}\p{Ll})/u', '$1-$2', $subject);

        // Insert hyphen on lower/digit -> upper transitions: "camelCase" -> "camel-Case", "ver2X" -> "ver-2-X"
        $s = self::safePregReplace('/(\p{Ll}|\p{Nd})(\p{Lu})/u', '$1-$2', $s);

        // Normalize any non-alphanumeric separators to a single hyphen
        $s = self::safePregReplace('/[^\p{L}\p{Nd}]+/u', '-', $s);

        // Trim leading/trailing hyphens
        $s = \trim($s, '-');

        if ('' === $s) {
            return '';
        }

        // Split into segments; ensure an array for static analyzers
        $parts = \preg_split('/-+/u', $s, -1, \PREG_SPLIT_NO_EMPTY);
        if (false === $parts) {
            $parts = [$s];
        }

        // Lowercase only segments that are not full-uppercase (preserve "A", "ID", "HTTP")
        foreach ($parts as $i => $p) {
            if (!\preg_match('/^\p{Lu}+$/u', $p)) {
                $parts[$i] = \mb_strtolower($p, 'UTF-8');
            }
        }

        return \implode('-', $parts);
    }

    /**
     * Safe wrapper around preg_replace that always returns a string.
     *
     * @psalm-param non-empty-string $pattern
     */
    private static function safePregReplace(string $pattern, string $replacement, string $subject): string
    {
        // If the pattern is invalid (including empty), preg_replace returns null.
        $result = \preg_replace($pattern, $replacement, $subject);

        if (!\is_string($result)) {
            throw new \InvalidArgumentException('Invalid PCRE pattern passed to preg_replace.');
        }

        return $result;
    }
}
