<?php

declare(strict_types=1);

namespace NoahVet\Reef\Command\Database;

use NoahVet\Reef\Command\Sync\AllSyncCommandInterface;
use NoahVet\Reef\Hasher\DirectoryHasherInterface;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\ArrayInput;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Process\Process;

#[AsCommand(
    name: 'reef:database:reset-fast',
    description: 'Load fixtures using a SQL cache if available.',
)]
class ResetFastDataBaseCommand extends Command implements ResetFastDataBaseCommandInterface
{
    public function __construct(
        protected readonly string $databaseUrl,
        protected readonly string $environment,
        protected readonly string $kernelProjectDir,
        protected readonly AllSyncCommandInterface $allSyncCommand,
        protected readonly DirectoryHasherInterface $directoryHasher,
    ) {
        parent::__construct();
    }

    public function __invoke(
        InputInterface $input,
        SymfonyStyle $io,
        OutputInterface $output,
    ): int {
        /** @var string[] $groups */
        $groups = $input->getOption('group') ?? [];
        // Compute all hashes in a single place (fixtures + migrations + sync + commands + global).
        $hashes = $this->directoryHasher->computeHashes($groups);

        $fixturesHash = $hashes['fixtures'];
        $fixtureCommandsHash = $hashes['fixtureCommands'];
        $migrationsHash = $hashes['migrations'];
        $syncHash = $hashes['sync'];
        $syncCommandsHash = $hashes['syncCommands'];
        $hash = $hashes['global'];

        // Shared cache for all Symfony environments.
        $cacheDir = $this->kernelProjectDir.'/var/cache/fixtures';
        $dumpPath = $cacheDir.'/'.$hash.'.sql';

        $io->writeln(\sprintf('Symfony env: <info>%s</info>', $this->environment));
        $io->writeln(\sprintf(
            'Fixture contexts (directories): <info>%s</info>',
            $groups ? \implode(', ', $groups) : '(all)',
        ));
        $io->writeln(\sprintf('Fixtures hash: <info>%s</info>', $fixturesHash));
        $io->writeln(\sprintf('Fixtures commands hash: <info>%s</info>', $fixtureCommandsHash));
        $io->writeln(\sprintf('Migrations hash: <info>%s</info>', $migrationsHash));
        $io->writeln(\sprintf('Sync data hash: <info>%s</info>', $syncHash));
        $io->writeln(\sprintf('Sync commands hash: <info>%s</info>', $syncCommandsHash));
        $io->writeln(\sprintf('Cache hash: <info>%s</info>', $hash));

        (new Filesystem())->mkdir($cacheDir);

        if (\is_file($dumpPath)) {
            $io->section('SQL dump found, importing...');

            // No drop / create here: this command assumes schema/db is already prepared.
            $this->importSqlDump(
                $dumpPath,
                $io,
            );

            $io->success('Database restored from SQL cache.');

            return Command::SUCCESS;
        }

        $io->section('No SQL cache for this fixture + schema + sync context, running migrations + sync + fixtures...');

        $this->runDoctrineMigrations($io);

        if (Command::FAILURE === $this->runAppSyncAll($output, $io)) {
            return Command::FAILURE;
        }

        $this->loadDoctrineFixtures(
            $io,
            $groups,
        );

        $io->section('Exporting database to SQL cache...');

        $this->exportDatabaseToSql(
            $dumpPath,
            $io,
        );

        $io->success('Fixtures loaded and SQL dump generated.');

        return Command::SUCCESS;
    }

    protected function configure(): void
    {
        $this
            ->addOption(
                'group',
                'g',
                InputOption::VALUE_REQUIRED | InputOption::VALUE_IS_ARRAY,
                'Fixture groups to load (and sub-directories to hash, e.g. pokemon, test)',
            )
        ;
    }

    private function runAppSyncAll(
        OutputInterface $output,
        SymfonyStyle $io,
    ): int {
        try {
            $this->allSyncCommand->run(
                new ArrayInput([]),
                $output,
            );
        } catch (\Throwable $exception) {
            $io->error(
                \sprintf(
                    "Failed to sync with %s!\nMessage: %s",
                    $this->allSyncCommand::class,
                    $exception->getMessage(),
                ),
            );

            return Command::FAILURE;
        }

        return Command::SUCCESS;
    }

    private function runDoctrineMigrations(SymfonyStyle $io): void
    {
        $cmd = [
            \PHP_BINARY,
            'bin/console',
            'doctrine:migrations:migrate',
            '-n',
        ];

        $process = new Process(
            $cmd,
            $this->kernelProjectDir,
        );

        $process->setTimeout(null);

        $process->run(
            static function (string $type, string $buffer) use ($io): void {
                $io->write($buffer);
            },
        );

        if (!$process->isSuccessful()) {
            $io->error("Doctrine migrations failed:\n".$process->getErrorOutput());

            return;
        }
    }

    /**
     * @param string[] $groups
     */
    private function loadDoctrineFixtures(
        SymfonyStyle $io,
        array $groups,
    ): void {
        $cmd = [
            \PHP_BINARY,
            'bin/console',
            'doctrine:fixtures:load',
            '-n',
            '--append',
        ];

        if ([] !== $groups) {
            $cmd[] = '--group';
            $cmd = [
                ...$cmd,
                ...$groups,
            ];
        }

        $process = new Process(
            $cmd,
            $this->kernelProjectDir,
        );

        $process->setTimeout(null);

        $process->run(
            static function (string $type, string $buffer) use ($io): void {
                $io->write($buffer);
            },
        );

        if (!$process->isSuccessful()) {
            $io->error("Fixtures failed:\n".$process->getErrorOutput());

            return;
        }
    }

    private function importSqlDump(
        string $dumpPath,
        SymfonyStyle $io,
    ): void {
        $io->writeln(\sprintf('Importing dump <comment>%s</comment>', $dumpPath));

        $config = $this->getDatabaseConfigFromUrl();

        $process = new Process([
            'mysql',
            '--host='.$config['host'],
            '--port='.$config['port'],
            '--user='.$config['user'],
            '--password='.$config['password'],
            $config['dbname'],
        ]);

        try {
            $contents = \file_get_contents($dumpPath);
            \assert(false !== $contents);
        } catch (\Exception $exception) {
            throw new \LogicException(
                "Error to get file from $dumpPath : {$exception->getMessage()}",
                previous: $exception,
            );
        }

        $process->setInput($contents);

        $process->mustRun();

        if (!$process->isSuccessful()) {
            $io->error('Error while importing SQL dump.');
            $io->error($process->getErrorOutput());

            return;
        }

        $io->writeln('Import finished.');
    }

    private function exportDatabaseToSql(
        string $dumpPath,
        SymfonyStyle $io,
    ): void {
        $io->writeln(\sprintf('Generating SQL dump to <comment>%s</comment>', $dumpPath));

        $config = $this->getDatabaseConfigFromUrl();

        $process = new Process([
            'mariadb-dump',
            '-h',
            $config['host'],
            '-u',
            $config['user'],
            '-p'.$config['password'],
            '--single-transaction',
            '--routines',
            '--triggers',
            '--events',
            $config['dbname'],
        ]);

        $process->mustRun();

        if (!$process->isSuccessful()) {
            $io->error('Error while generating SQL dump.');
            $io->error($process->getErrorOutput());

            return;
        }

        \file_put_contents(
            $dumpPath,
            $process->getOutput(),
        );

        $io->writeln('SQL dump generated.');
    }

    /**
     * @return array{
     *     host: string,
     *     port: string,
     *     user: string,
     *     password: string,
     *     dbname: string
     * }
     */
    private function getDatabaseConfigFromUrl(): array
    {
        // We know the DatabaseUrl pattern in .env, symfony standard
        /** @var array{
         *     fragment?: string,
         *     host?: string,
         *     pass?: string,
         *     path?: string,
         *     port?: int,
         *     query?: string,
         *     scheme?: string,
         *     user?: string
         * }|false $parts
         */
        $parts = \parse_url($this->databaseUrl);

        if (false === $parts) {
            throw new \RuntimeException('Invalid DATABASE_URL: parse_url() returned false.');
        }

        $host = ($parts['host'] ?? '127.0.0.1');
        $port = isset($parts['port']) ? (string) $parts['port'] : '3306';
        $user = ($parts['user'] ?? 'root');
        $password = ($parts['pass'] ?? 'root');
        $path = ($parts['path'] ?? '');
        $dbname = \ltrim($path, '/') ?: 'cam';

        return [
            'host' => $host,
            'port' => $port,
            'user' => $user,
            'password' => $password,
            'dbname' => $dbname,
        ];
    }
}
