<?php

declare(strict_types=1);

namespace NoahVet\Reef\Jane\Endpoint;

class UpdateOneCreditNote extends \NoahVet\Reef\Jane\Runtime\Client\BaseEndpoint implements \NoahVet\Reef\Jane\Runtime\Client\Endpoint
{
    use \NoahVet\Reef\Jane\Runtime\Client\EndpointTrait;

    protected $id;

    /**
     * Update a Credit Note.
     *
     * @param string $id               the resource ID
     * @param array  $headerParameters {
     *
     * @var string $Accept-Language The current locale.
     *             }
     */
    public function __construct(string $id, \NoahVet\Reef\Jane\Model\SchemasCreditNoteRequest $requestBody, array $headerParameters = [])
    {
        $this->id = $id;
        $this->body = $requestBody;
        $this->headerParameters = $headerParameters;
    }

    public function getMethod(): string
    {
        return 'PATCH';
    }

    public function getUri(): string
    {
        return \str_replace(['{id}'], [$this->id], '/v1/credit_notes/{id}');
    }

    public function getBody(\Symfony\Component\Serializer\SerializerInterface $serializer, $streamFactory = null): array
    {
        if ($this->body instanceof \NoahVet\Reef\Jane\Model\SchemasCreditNoteRequest) {
            return [['Content-Type' => ['application/json']], $serializer->serialize($this->body, 'json')];
        }

        return [[], null];
    }

    public function getExtraHeaders(): array
    {
        return ['Accept' => ['application/json']];
    }

    public function getAuthenticationScopes(): array
    {
        return ['BearerAuth'];
    }

    protected function getHeadersOptionsResolver(): \Symfony\Component\OptionsResolver\OptionsResolver
    {
        $optionsResolver = parent::getHeadersOptionsResolver();
        $optionsResolver->setDefined(['Accept-Language']);
        $optionsResolver->setRequired([]);
        $optionsResolver->setDefaults([]);
        $optionsResolver->addAllowedTypes('Accept-Language', ['string']);

        return $optionsResolver;
    }

    /**
     * @return \NoahVet\Reef\Jane\Model\CreditNoteResponse
     *
     * @throws \NoahVet\Reef\Jane\Exception\UpdateOneCreditNoteBadRequestException
     * @throws \NoahVet\Reef\Jane\Exception\UpdateOneCreditNoteNotFoundException
     * @throws \NoahVet\Reef\Jane\Exception\UnexpectedStatusCodeException
     */
    protected function transformResponseBody(\Psr\Http\Message\ResponseInterface $response, \Symfony\Component\Serializer\SerializerInterface $serializer, ?string $contentType = null)
    {
        $status = $response->getStatusCode();
        $body = (string) $response->getBody();
        if ((null === $contentType) === false && (200 === $status && false !== \mb_strpos($contentType, 'application/json'))) {
            return $serializer->deserialize($body, 'NoahVet\\Reef\\Jane\\Model\\CreditNoteResponse', 'json');
        }
        if (400 === $status) {
            throw new \NoahVet\Reef\Jane\Exception\UpdateOneCreditNoteBadRequestException($response);
        }
        if (404 === $status) {
            throw new \NoahVet\Reef\Jane\Exception\UpdateOneCreditNoteNotFoundException($response);
        }
        throw new \NoahVet\Reef\Jane\Exception\UnexpectedStatusCodeException($status, $body);
    }
}
