<?php

declare(strict_types=1);

namespace NoahVet\Reef\Jane\Endpoint;

class PutResourceItem extends \NoahVet\Reef\Jane\Runtime\Client\BaseEndpoint implements \NoahVet\Reef\Jane\Runtime\Client\Endpoint
{
    use \NoahVet\Reef\Jane\Runtime\Client\EndpointTrait;

    protected $slug;

    protected $accept;

    /**
     * Replaces the Resource resource.
     *
     * @param string $slug   Resource identifier
     * @param array  $accept Accept content header application/json|text/html
     */
    public function __construct(string $slug, \NoahVet\Reef\Jane\Model\Resource $requestBody, array $accept = [])
    {
        $this->slug = $slug;
        $this->body = $requestBody;
        $this->accept = $accept;
    }

    public function getMethod(): string
    {
        return 'PUT';
    }

    public function getUri(): string
    {
        return \str_replace(['{slug}'], [$this->slug], '/api/resources/{slug}');
    }

    public function getBody(\Symfony\Component\Serializer\SerializerInterface $serializer, $streamFactory = null): array
    {
        if ($this->body instanceof \NoahVet\Reef\Jane\Model\Resource) {
            return [['Content-Type' => ['application/json']], $serializer->serialize($this->body, 'json')];
        }
        if ($this->body instanceof \NoahVet\Reef\Jane\Model\Resource) {
            return [['Content-Type' => ['text/html']], $this->body];
        }

        return [[], null];
    }

    public function getExtraHeaders(): array
    {
        return ['Accept' => ['application/json']];
    }

    public function getAuthenticationScopes(): array
    {
        return ['BearerAuth'];
    }

    /**
     * @return \NoahVet\Reef\Jane\Model\Resource
     *
     * @throws \NoahVet\Reef\Jane\Exception\PutResourceItemBadRequestException
     * @throws \NoahVet\Reef\Jane\Exception\PutResourceItemUnprocessableEntityException
     * @throws \NoahVet\Reef\Jane\Exception\PutResourceItemNotFoundException
     * @throws \NoahVet\Reef\Jane\Exception\UnexpectedStatusCodeException
     */
    protected function transformResponseBody(\Psr\Http\Message\ResponseInterface $response, \Symfony\Component\Serializer\SerializerInterface $serializer, ?string $contentType = null)
    {
        $status = $response->getStatusCode();
        $body = (string) $response->getBody();
        if ((null === $contentType) === false && (200 === $status && false !== \mb_strpos($contentType, 'application/json'))) {
            return $serializer->deserialize($body, 'NoahVet\\Reef\\Jane\\Model\\Resource', 'json');
        }
        if (400 === $status) {
            throw new \NoahVet\Reef\Jane\Exception\PutResourceItemBadRequestException($response);
        }
        if (422 === $status) {
            throw new \NoahVet\Reef\Jane\Exception\PutResourceItemUnprocessableEntityException($response);
        }
        if (404 === $status) {
            throw new \NoahVet\Reef\Jane\Exception\PutResourceItemNotFoundException($response);
        }
        throw new \NoahVet\Reef\Jane\Exception\UnexpectedStatusCodeException($status, $body);
    }
}
