<?php

declare(strict_types=1);

namespace NoahVet\Reef\Jane\Endpoint;

class GetPermissionsResourceCollection extends \NoahVet\Reef\Jane\Runtime\Client\BaseEndpoint implements \NoahVet\Reef\Jane\Runtime\Client\Endpoint
{
    use \NoahVet\Reef\Jane\Runtime\Client\EndpointTrait;

    /**
     * Retrieve effective permissions on a collection of resources.
     *
     * @param array $queryParameters {
     *
     * @var array  $slug[] List of resources you want permissions for
     * @var string $principal Principal ID (if different from the one doing the query)
     * @var int    $page The collection page number
     * @var int    $itemsPerPage The number of items per page
     * @var string $slug
     *             }
     *
     * @param array $headerParameters {
     *
     * @var string $X-Reef-Client-ID Public OAuth Client ID
     * @var string $X-Reef-Client-HMAC HMAC computed using hash_hmac('sha256', bearer, clientSecret)
     *             }
     */
    public function __construct(array $queryParameters = [], array $headerParameters = [])
    {
        $this->queryParameters = $queryParameters;
        $this->headerParameters = $headerParameters;
    }

    public function getMethod(): string
    {
        return 'GET';
    }

    public function getUri(): string
    {
        return '/api/resources/permissions';
    }

    public function getBody(\Symfony\Component\Serializer\SerializerInterface $serializer, $streamFactory = null): array
    {
        return [[], null];
    }

    public function getExtraHeaders(): array
    {
        return ['Accept' => ['application/json']];
    }

    public function getAuthenticationScopes(): array
    {
        return ['BearerAuth'];
    }

    protected function getQueryOptionsResolver(): \Symfony\Component\OptionsResolver\OptionsResolver
    {
        $optionsResolver = parent::getQueryOptionsResolver();
        $optionsResolver->setDefined(['slug', 'principal', 'page', 'itemsPerPage']);
        $optionsResolver->setRequired(['slug']);
        $optionsResolver->setDefaults(['page' => 1, 'itemsPerPage' => 30]);
        $optionsResolver->addAllowedTypes('slug', ['array']);
        $optionsResolver->addAllowedTypes('principal', ['string']);
        $optionsResolver->addAllowedTypes('page', ['int']);
        $optionsResolver->addAllowedTypes('itemsPerPage', ['int']);
        $optionsResolver->addAllowedTypes('slug', ['string']);

        return $optionsResolver;
    }

    protected function getHeadersOptionsResolver(): \Symfony\Component\OptionsResolver\OptionsResolver
    {
        $optionsResolver = parent::getHeadersOptionsResolver();
        $optionsResolver->setDefined(['X-Reef-Client-ID', 'X-Reef-Client-HMAC']);
        $optionsResolver->setRequired([]);
        $optionsResolver->setDefaults([]);
        $optionsResolver->addAllowedTypes('X-Reef-Client-ID', ['string']);
        $optionsResolver->addAllowedTypes('X-Reef-Client-HMAC', ['string']);

        return $optionsResolver;
    }

    /**
     * @return \NoahVet\Reef\Jane\Model\PermissionGrantList
     *
     * @throws \NoahVet\Reef\Jane\Exception\UnexpectedStatusCodeException
     */
    protected function transformResponseBody(\Psr\Http\Message\ResponseInterface $response, \Symfony\Component\Serializer\SerializerInterface $serializer, ?string $contentType = null)
    {
        $status = $response->getStatusCode();
        $body = (string) $response->getBody();
        if ((null === $contentType) === false && (200 === $status && false !== \mb_strpos($contentType, 'application/json'))) {
            return $serializer->deserialize($body, 'NoahVet\\Reef\\Jane\\Model\\PermissionGrantList', 'json');
        }
        throw new \NoahVet\Reef\Jane\Exception\UnexpectedStatusCodeException($status, $body);
    }
}
