<?php

declare(strict_types=1);

namespace NoahVet\Reef\Test\A_Unit\Security;

use NoahVet\Reef\Factory\ClientFactory;
use NoahVet\Reef\Jane\Client;
use NoahVet\Reef\Jane\Model\UserinfoGetResponse200;
use NoahVet\Reef\Security\ReefOAuthAuthenticator;
use NoahVet\Reef\Security\User\ReefOAuthUser;
use PHPUnit\Framework\TestCase;
use Psr\Cache\CacheItemInterface;
use Psr\Cache\CacheItemPoolInterface;

class ReefOAuthAuthenticatorTest extends TestCase
{
    public function testLoadUserNoCache(): void
    {
        $clientFactory = $this->getMockBuilder(ClientFactory::class)
            ->disableOriginalConstructor()
            ->getMock()
        ;

        $client = $this->getMockBuilder(Client::class)
            ->disableOriginalConstructor()
            ->getMock()
        ;

        $userInfo = new UserinfoGetResponse200();

        $userInfo
            ->setIdentifier('user-id')
            ->setEmail('e@mail.com')
        ;

        $clientFactory
            ->method('create')
            ->with('bearer')
            ->willReturn($client)
        ;

        $client->method('userinfo')
            ->willReturn($userInfo)
        ;

        $authenticator = new ReefOAuthAuthenticator($clientFactory);

        $user = $authenticator->loadUser('bearer');
        self::assertInstanceOf(ReefOAuthUser::class, $user);
        self::assertEquals('reef:principal:user-id', $user->getUserIdentifier());
        self::assertSame($userInfo, $user->getUserinfo());
    }

    public function testLoadUserCached(): void
    {
        $clientFactory = $this->getMockBuilder(ClientFactory::class)
            ->disableOriginalConstructor()
            ->getMock()
        ;

        $cache = $this->getMockBuilder(CacheItemPoolInterface::class)
            ->disableOriginalConstructor()
            ->getMock()
        ;

        $user = new ReefOAuthUser('user-id');

        $cacheItem = $this->getMockBuilder(CacheItemInterface::class)
            ->disableOriginalConstructor()
            ->getMock()
        ;

        $cacheItem->method('isHit')
            ->willReturn(true)
        ;

        $cacheItem->method('get')
            ->willReturn($user)
        ;

        $cache
            ->expects($this->once())
            ->method('getItem')
            ->with('userinfo-bearer')
            ->willReturn($cacheItem)
        ;

        $authenticator = new ReefOAuthAuthenticator($clientFactory);
        $authenticator->setCache($cache);

        self::assertSame($user, $authenticator->loadUser('bearer'));
    }

    public function testLoadUserNotCached(): void
    {
        $clientFactory = $this->getMockBuilder(ClientFactory::class)
            ->disableOriginalConstructor()
            ->getMock()
        ;

        $client = $this->getMockBuilder(Client::class)
            ->disableOriginalConstructor()
            ->getMock()
        ;

        $userInfo = new UserinfoGetResponse200();

        $userInfo
            ->setIdentifier('user-id')
            ->setEmail('e@mail.com')
        ;

        $clientFactory
            ->method('create')
            ->with('bearer')
            ->willReturn($client)
        ;

        $client->method('userinfo')
            ->willReturn($userInfo)
        ;

        $cache = $this->getMockBuilder(CacheItemPoolInterface::class)
            ->disableOriginalConstructor()
            ->getMock()
        ;

        $cacheItem = $this->getMockBuilder(CacheItemInterface::class)
            ->disableOriginalConstructor()
            ->getMock()
        ;

        $cacheItem->method('isHit')
            ->willReturn(false)
        ;

        $cache
            ->expects($this->once())
            ->method('getItem')
            ->with('userinfo-bearer')
            ->willReturn($cacheItem)
        ;

        $cache
            ->expects($this->once())
            ->method('save')
            ->with($cacheItem)
        ;

        $authenticator = new ReefOAuthAuthenticator($clientFactory);
        $authenticator->setCache($cache);

        $user = $authenticator->loadUser('bearer');
        self::assertInstanceOf(ReefOAuthUser::class, $user);
        self::assertEquals('reef:principal:user-id', $user->getUserIdentifier());
        self::assertSame($userInfo, $user->getUserinfo());
    }
}
