<?php

declare(strict_types=1);

namespace NoahVet\Reef\Test\A_Unit\Domain\Tool;

use NoahVet\Reef\Domain\Tool\DateTool;
use NoahVet\Reef\Domain\Tool\DateToolInterface;
use PHPUnit\Framework\TestCase;

final class DateToolTest extends TestCase
{
    private DateToolInterface $subject;

    protected function setUp(): void
    {
        $this->subject = new DateTool();
    }

    public function testFromNullableStringToDateTime(): void
    {
        $dateTime = new \DateTime('2022-12-31T00:00:00+00:00');

        $result = $this->subject::fromNullableStringToDateTime(
            '2022-12-31T00:00:00+00:00',
        );

        self::assertEquals(
            $dateTime,
            $result,
        );
    }

    public function testFromNullableStringToDateTimeWhenDateIsEqualsToNull(): void
    {
        self::assertNull(
            $this->subject::fromNullableStringToDateTime(null),
        );
    }

    public function testToFirstSecondOfTheDayWithNegativeUTC(): void
    {
        $dateTime = new \DateTime('2021-12-31T00:00:00-02:00');
        $this->subject::toFirstSecondOfTheDay($dateTime);

        self::assertEquals(
            '2021-12-31T00:00:00-02:00',
            $dateTime->format(\DATE_ATOM),
        );
    }

    public function testToFirstSecondOfTheDayWithPositiveUTC(): void
    {
        $dateTime = new \DateTime('2021-12-31T00:00:00+02:00');
        $this->subject::toFirstSecondOfTheDay($dateTime);

        self::assertEquals(
            '2021-12-31T00:00:00+02:00',
            $dateTime->format(\DATE_ATOM),
        );
    }

    public function testToFirstSecondOfTheDayWithUTCZero(): void
    {
        $dateTime = new \DateTime('2022-12-31T00:00:00+00:00');
        $this->subject::toFirstSecondOfTheDay($dateTime);

        self::assertEquals(
            '2022-12-31T00:00:00+00:00',
            $dateTime->format(\DATE_ATOM),
        );
    }

    public function testToLastSecondOfTheDayWithNegativeUTC(): void
    {
        $dateTime = new \DateTime('2021-12-31T00:00:00-02:00');
        $this->subject::toLastSecondOfTheDay($dateTime);

        self::assertEquals(
            '2021-12-31T23:59:59-02:00',
            $dateTime->format(\DATE_ATOM),
        );
    }

    public function testToLastSecondOfTheDayWithPositiveUTC(): void
    {
        $dateTime = new \DateTime('2021-12-31T00:00:00+02:00');
        $this->subject::toLastSecondOfTheDay($dateTime);

        self::assertEquals(
            '2021-12-31T23:59:59+02:00',
            $dateTime->format(\DATE_ATOM),
        );
    }

    public function testToLastSecondOfTheDayWithUTCZero(): void
    {
        $dateTime = new \DateTime('2022-12-31T00:00:00+00:00');
        $this->subject::toLastSecondOfTheDay($dateTime);

        self::assertEquals(
            '2022-12-31T23:59:59+00:00',
            $dateTime->format(\DATE_ATOM),
        );
    }

    public function testFromStringToDateTimeNotImmutable(): void
    {
        $dateTime = new \DateTime('2022-12-31T00:00:00+00:00');

        $result = $this->subject::fromStringToDateTime(
            '2022-12-31T00:00:00+00:00',
            \DateTimeInterface::ATOM,
            $this->subject::UTC,
            false,
        );

        self::assertEquals(
            $dateTime,
            $result,
        );
    }

    public function testFromStringToDateTimeWhenDateStringIsNotADate(): void
    {
        $this->expectException(\LogicException::class);
        $this->expectExceptionMessage('The $dateString parameter is not in '.\DateTimeInterface::ATOM.' format.');
        $this->subject::fromStringToDateTime('bad');
    }
}
