<?php

declare(strict_types=1);

namespace NoahVet\Reef\Security;

use NoahVet\Reef\Factory\ClientFactory;
use NoahVet\Reef\Jane\Exception\UserinfoUnauthorizedException;
use NoahVet\Reef\Jane\Model\UserinfoGetResponse200;
use NoahVet\Reef\Security\Authentication\Token\ReefOAuthToken;
use NoahVet\Reef\Security\User\ReefOAuthUser;
use Psr\Cache\CacheItemPoolInterface;
use Psr\Cache\InvalidArgumentException;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\Security\Http\Authenticator\AbstractAuthenticator;
use Symfony\Component\Security\Http\Authenticator\Passport\Badge\UserBadge;
use Symfony\Component\Security\Http\Authenticator\Passport\Passport;
use Symfony\Component\Security\Http\Authenticator\Passport\SelfValidatingPassport;

class ReefOAuthAuthenticator extends AbstractAuthenticator implements ReefOAuthAuthenticatorInterface
{
    /**
     * Default : cache user for 30 seconds.
     */
    public const REEF_USER_CACHE_DURATION = 30;

    protected ClientFactory $clientFactory;

    protected ?CacheItemPoolInterface $cache = null;

    public function __construct(ClientFactory $clientFactory)
    {
        $this->clientFactory = $clientFactory;
    }

    public function supports(Request $request): ?bool
    {
        return $request->headers->has('Authorization')
            && \str_starts_with($request->headers->get('Authorization') ?? '', 'Bearer ');
    }

    public function authenticate(Request $request): Passport
    {
        $oauthToken = \trim(\substr($request->headers->get('Authorization') ?? '', 7));

        return new SelfValidatingPassport(
            new UserBadge($oauthToken, [$this, 'loadUser']),
        );
    }

    public function createToken(Passport $passport, string $firewallName): TokenInterface
    {
        /** @var UserBadge $userBadge */
        $userBadge = $passport->getBadge(UserBadge::class);
        /** @var ReefOAuthUser $user */
        $user = $userBadge->getUser();

        return new ReefOAuthToken($firewallName, $userBadge->getUserIdentifier(), $user);
    }

    public function onAuthenticationSuccess(Request $request, TokenInterface $token, string $firewallName): ?Response
    {
        return null;
    }

    public function onAuthenticationFailure(Request $request, AuthenticationException $exception): ?Response
    {
        if (\str_starts_with($request->headers->get('Authorization') ?? '', 'Bearer ')) {
            return new JsonResponse(['error' => 'Invalid bearer token provided'], Response::HTTP_FORBIDDEN);
        } else {
            return new JsonResponse(['error' => 'No bearer token provided'], Response::HTTP_UNAUTHORIZED);
        }
    }

    public function loadUser(string $bearerToken): ?ReefOAuthUser
    {
        try {
            $cacheItem = $this->cache?->getItem('userinfo-'.$bearerToken);
        } catch (InvalidArgumentException) {
            $cacheItem = null;
        }

        if ($cacheItem && $cacheItem->isHit()) {
            $cachedUser = $cacheItem->get();

            if (null === $cachedUser || $cachedUser instanceof ReefOAuthUser) {
                return $cachedUser;
            }
        }

        $client = $this->clientFactory->create($bearerToken);

        try {
            /** @var UserinfoGetResponse200|null $userInfo */
            $userInfo = $client->userinfo();

            if (!$userInfo) {
                $ret = null;
            } else {
                $ret = ReefOAuthUser::createFromUserinfo($userInfo);
            }

            if ($this->cache && $cacheItem) {
                $cacheItem->set($ret);
                $cacheItem->expiresAfter(self::REEF_USER_CACHE_DURATION);
                $this->cache->save($cacheItem);
            }

            return $ret;
        } catch (UserinfoUnauthorizedException) {
            // Invalid token
            return null;
        }
    }

    public function setCache(?CacheItemPoolInterface $cache): void
    {
        $this->cache = $cache;
    }

    public function getCache(): ?CacheItemPoolInterface
    {
        return $this->cache;
    }
}
