<?php

declare(strict_types=1);

namespace NoahVet\Reef\RabbitMQ\Event;

use NoahVet\Reef\RabbitMQ\Task\ConsoleCommandTask;
use NoahVet\Reef\RabbitMQ\Task\Executor\ConsoleTaskCommandExecutorFactoryInterface;
use NoahVet\Reef\RabbitMQ\Task\InternalTaskInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\Serializer\SerializerInterface;

class DefaultInternalTaskEventSubscriber implements EventSubscriberInterface
{
    public function __construct(
        private readonly ConsoleTaskCommandExecutorFactoryInterface $consoleTaskCommandExecutorFactory,
        private readonly LoggerInterface $logger,
        private readonly SerializerInterface $serializer,
    ) {
    }

    /**
     * @return array<string, string>
     */
    public static function getSubscribedEvents(): array
    {
        return [
            AMQPMessageEvent::class => 'onAMQPMessage',
            InternalTaskEvent::class => 'onInternalTask',
        ];
    }

    public function onAMQPMessage(AMQPMessageEvent $event): void
    {
        $message = $event->message;

        $contentEncoding = $message->get_properties()['content_encoding'] ?? $message->getContentEncoding();

        if ('application/json' !== $contentEncoding) {
            return;
        }
        try {
            /* @var InternalTaskInterface $internalTask */
            $event->internalTask = $this->serializer->deserialize(
                $message->getBody(),
                InternalTaskInterface::class,
                'json',
            );
        } catch (\Exception $e) {
            $this->logger->error('Failed to deserialize AMQP Message', [
                'exception' => $e,
                'message' => $message->getBody(),
            ]);
        }
    }

    public function onInternalTask(InternalTaskEvent $event): void
    {
        $internalTask = $event->internalTask;
        if (!$internalTask instanceof ConsoleCommandTask) {
            return;
        }

        $executor = $this->consoleTaskCommandExecutorFactory->create($internalTask);

        $this->logger->debug(
            'Executing internal command task',
            [
                'command' => $internalTask->command->getName(),
                'arguments' => $internalTask->arguments,
                'executor' => $executor::class,
            ],
        );

        $retCode = $executor->execute($internalTask);

        $this->logger->debug(
            'Internal task return value',
            [
                'command' => $internalTask->command->getName(),
                'arguments' => $internalTask->arguments,
                'retCode' => $retCode,
            ],
        );
    }
}
