<?php

declare(strict_types=1);

namespace NoahVet\Reef\RabbitMQ\Consumer;

use NoahVet\Reef\RabbitMQ\Event\AMQPMessageEvent;
use NoahVet\Reef\RabbitMQ\Event\InternalTaskEvent;
use NoahVet\Reef\RabbitMQ\InternalQueueTrait;
use PhpAmqpLib\Channel\AMQPChannel;
use PhpAmqpLib\Connection\AbstractConnection;
use PhpAmqpLib\Message\AMQPMessage;
use Psr\Log\LoggerInterface;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

class InternalTaskConsumer implements InternalTaskConsumerInterface
{
    use InternalQueueTrait;

    private AMQPChannel $channel;

    public function __construct(
        AbstractConnection $connection,
        private readonly EventDispatcherInterface $eventDispatcher,
        private readonly LoggerInterface $logger,
        private readonly string $reefAppName,
    ) {
        $this->channel = $connection->channel();
        $this->declareQueue($this->channel, $this->reefAppName);

        $this->channel->basic_consume(
            'reef:'.$this->reefAppName,
            'reef-internal',
            false,
            false,
            false,
            false,
            [$this, 'onRawMessage'],
        );

        $this->channel->consume();
    }

    public function onRawMessage(AMQPMessage $message): void
    {
        try {
            $this->logger->debug(
                'Received AMQP Message',
                [
                    'properties' => $message->get_properties(),
                    'body' => $message->getBody(),
                ],
            );

            $messageEvent = new AMQPMessageEvent($message);
            $this->eventDispatcher->dispatch($messageEvent);

            if (!$messageEvent->internalTask) {
                $this->logger->warning(
                    'Unable to deserialize message body to InternalTaskInterface',
                    [
                        'body' => $message->getBody(),
                    ],
                );

                $message->nack();

                return;
            }

            $this->logger->debug(
                'Found internal task',
                [
                    'class' => \get_class($messageEvent->internalTask),
                ],
            );

            $this->eventDispatcher->dispatch(new InternalTaskEvent($messageEvent->internalTask));

            $message->ack();
        } catch (\Exception $e) {
            $this->logger->error(
                $e->getMessage(),
                [
                    'file' => $e->getFile(),
                    'line' => $e->getLine(),
                ],
            );

            $message->nack();
        }
    }

    public function consume(float $timeout = 10.0): void
    {
        $this->channel->consume($timeout);
    }

    public function isOpen(): bool
    {
        return $this->channel->is_open();
    }

    public function stopConsume(): void
    {
        $this->channel->stopConsume();
        $this->channel->close();
    }
}
