<?php

declare(strict_types=1);

namespace NoahVet\Reef\Business\Enumerator;

use NoahVet\Reef\Jane\Client;
use NoahVet\Reef\Jane\Exception\GetAllManufacturerUserBadRequestException;
use NoahVet\Reef\Jane\Model\V1ManufacturersGetResponse200Item as Manuf200;
use NoahVet\Reef\Jane\Model\V1ManufacturersGetResponse206Item as Manuf206;
use NoahVet\Reef\Jane\Model\V1ManufacturerUsersGetResponse200Item as ManufUser200;
use NoahVet\Reef\Jane\Model\V1ManufacturerUsersGetResponse206Item as ManufUser206;

class ManufacturerTeamMembersEnumerator
{
    public function __construct(
        protected Client $client,
    ) {
    }

    /**
     * @param ?string[] $manufacturerIdFilter
     *
     * @return \Generator<ManufUser200|ManufUser206>
     */
    public function enumerateAll(array $manufacturerIdFilter = null, int $querySize = 50): \Generator
    {
        for ($page = 0; true; ++$page) {
            try {
                $params = ['page' => $page, 'limit' => $querySize];
                if ($manufacturerIdFilter) {
                    $params['manufacturerId'] = $manufacturerIdFilter;
                }
                $teamMembers = $this->client->getAllManufacturerUser($params);

                \assert(\is_array($teamMembers));

                foreach ($teamMembers as $teamMember) {
                    yield $teamMember;
                }

                if (\count($teamMembers) < $querySize) {
                    break;
                }
            } catch (GetAllManufacturerUserBadRequestException) {
                // End of enumeration
                break;
            }
        }
    }

    /**
     * @param Manuf200[]|Manuf206[] $manufacturers
     *
     * @return array<string, array<ManufUser200|ManufUser206>>
     */
    public function getTeamMembers(array $manufacturers, int $querySize = 50): array
    {
        $idFilter = \array_map(fn ($m): string => $m->getId(), $manufacturers);

        return \array_reduce(
            \iterator_to_array($this->enumerateAll($idFilter, $querySize)),
            function (array $carry, ManufUser200|ManufUser206 $item): array {
                $manufacturerId = $item->getManufacturer()?->getId();

                if (null !== $manufacturerId) {
                    $carry[$manufacturerId][] = $item;
                }

                return $carry;
            },
            [],
        );
    }
}
